<?php

	namespace PageEditor;

	use Nox\ORM\ColumnQuery;
	use Page\Page;
	use Page\PageType;
	use Page\PublicationStatus;
	use PageEditor\Exceptions\NoPageFoundWithID;
	use PageEditor\Exceptions\PageLayoutDoesntExist;
	use PageEditor\Exceptions\PageLayoutIsBlank;
	use PageEditor\Exceptions\PageNameExists;
	use PageEditor\Exceptions\PageNameExistsWithSamePageType;
	use PageEditor\Exceptions\PageNameIsBlank;
	use PageEditor\Exceptions\PageRouteInUse;
	use PageEditor\Exceptions\PageRouteIsBlank;
	use PageEditor\Exceptions\PublicationStatusDoesntExist;
	use System\Layouts;
	use Uplift\Exceptions\MalformedValue;
	use ValueError;

	/**
	 * Utility methods to validate page properties before saved or created.
	 */
	class PageEditorPageValidationService{

		/**
		 * @throws PageNameExistsWithSamePageType
		 * @throws PageNameIsBlank
		 * @throws PageRouteIsBlank
		 * @throws PageRouteInUse
		 * @throws PageLayoutIsBlank
		 * @throws PageLayoutDoesntExist
		 * @throws PublicationStatusDoesntExist
		 * @throws MalformedValue
		 */
		public static function validateCommonProperties(
			Page $page,
			string $pageName,
			string $pageRoute,
			string $pageLayout,
			int $publicationStatus,
		): void{
			// Test if page name is blank
			if (strlen(trim($pageName)) === 0){
				throw new PageNameIsBlank("Page name cannot be blank.");
			}

			// Check if page name is in use by another page with the same page type
			/** @var Page | null $existingPageWithSameNameInSamePageType */
			$existingPageWithSameNameInSamePageType = Page::queryOne(
				columnQuery: (new ColumnQuery())
					->where("pageType","=",$page->pageType)
					->and()
					->where("id","!=",$page->id)
					->and()
					->where("pageName","=",trim($pageName)),
			);

			if ($existingPageWithSameNameInSamePageType !== null){
				throw new PageNameExistsWithSamePageType("A page with the name {$pageName} already exists in the category {$page->pageType}.");
			}

			// Test the page route is not blank
			if (strlen(trim($pageRoute)) === 0){
				throw new PageRouteIsBlank("The page URI/route cannot be blank.");
			}

			// Check if the page route starts with a / - it must start with a forward slash
			if (!str_starts_with(haystack: $pageRoute, needle: "/")){
				throw new MalformedValue("Page routes must begin with a forward slash - /");
			}

			// Check if _any_ other page in the system has that same route. This cannot happen for obvious
			// reasons.
			/** @var Page | null $existingPageWithSameRoute */
			$existingPageWithSameRoute = Page::queryOne(
				columnQuery: (new ColumnQuery())
					->where("pageRoute","=",$page->pageType)
					->and()
					->where("id","!=",$page->id),
			);

			if ($existingPageWithSameRoute !== null){
				throw new PageRouteInUse("A page with that route already exists. The route is currently in use by page {$existingPageWithSameRoute->pageName} with ID {$existingPageWithSameRoute->id}.");
			}

			// Layout cannot be blank
			if (strlen($pageLayout) === 0){
				throw new PageLayoutIsBlank("The layout for the page cannot be blank.");
			}

			// Check if the layout in use exists in the system
			if (Layouts::isLayoutAvailable($pageLayout) === false){
				throw new PageLayoutDoesntExist("This page has a layout named {$pageLayout} selected, but that layout does not exist in the system's theme.");
			}

			// Check if the publication status exists as an enum
			$doesEnumExist = false;
			foreach(PublicationStatus::cases() as $status){
				if ($publicationStatus === $status->value){
					$doesEnumExist = true;
					break;
				}
			}

			if (!$doesEnumExist){
				throw new PublicationStatusDoesntExist("No enum found with string name {$publicationStatus}.");
			}
		}
	}