<?php

	namespace PageRestoration;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;
	use System\HttpHelper;
	use Uplift\Exceptions\NoObjectFound;

	#[Controller]
	#[RouteBase("/uplift/deleted-page-restoration")]
	class PageRestorationController extends BaseController
	{

		#[Route("GET", "/")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::VIEW_PAGE_HISTORY)]
		public function mainView(): string
		{
			return Renderer::renderView(
				viewFileName: "page-restoration/main.php",
			);
		}

		#[Route("GET", "/pages")]
		#[RequirePermission(PermissionCategories::VIEW_PAGE_HISTORY)]
		#[UseJSON]
		public function getDeletedPages(Request $request): JSONResult
		{
			$pageNumber = (int) $request->getQueryValue("pageNumber");
			$query = $request->getQueryValue("query");

			$queryResult = PageRestorationService::queryDeletedPages(
				pageNumber:$pageNumber,
				query:$query,
			);

			return new JSONSuccess($queryResult);
		}

		#[Route("POST", "/restore-page")]
		#[RequirePermission(PermissionCategories::VIEW_PAGE_HISTORY)]
		#[UseJSON]
		#[ProcessRequestBody]
		public function restorePageFromLatestArchive(Request $request): JSONResult
		{
			$payload = $request->getPayload();

			try{
				$rowID = $payload->getTextPayload("archive-row-id");
			}catch(NoPayloadFound $e){
				http_response_code(400);
				return new JSONError($e->getMessage());
			}

			try {
				PageRestorationService::restorePageFromArchive(
					archiveID: (int)$rowID->contents
				);
			}catch(NoObjectFound $e){
				http_response_code(500);
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::DELETED_PAGE_RESTORED->value,
				accountID: (Account::getCurrentUser())->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"editArchiveID"=>$rowID->contents,
				]),
			);

			return new JSONSuccess();
		}

	}