<?php

	namespace SenzuImporter;

	use APIPublic\ControlPanelAPIException;
	use GuzzleHttp\Client;
	use GuzzleHttp\Exception\GuzzleException;
	use GuzzleHttp\Exception\RequestException;
	use GuzzleHttp\RequestOptions;
	use MonologWrapper\MonologWrapper;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Exceptions\NoPrimaryKey;
	use Page\Page;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use PageArchives\PageArchivesService;
	use PageEditor\Exceptions\PageLayoutDoesntExist;
	use PageEditor\Exceptions\PageLayoutIsBlank;
	use PageEditor\Exceptions\PageNameExists;
	use PageEditor\Exceptions\PageNameExistsWithSamePageType;
	use PageEditor\Exceptions\PageNameIsBlank;
	use PageEditor\Exceptions\PageRouteInUse;
	use PageEditor\Exceptions\PageRouteIsBlank;
	use PageEditor\Exceptions\PublicationStatusDoesntExist;
	use PageEditor\PageEditorService;
	use Settings\Setting;
	use Settings\Settings;
	use System\Layouts;
	use System\System;
	use TemplateManager\PageLayouts\PageLayoutSection;
	use TemplateManager\PageLayouts\PageLayoutSectionsProvider;
	use Uplift\Exceptions\IncompatiblePageType;
	use Uplift\Exceptions\MalformedValue;

	class SenzuImporterService
	{
		/**
		 *
		 * @throws ControlPanelAPIException
		 * @throws MalformedValue
		 * @throws GuzzleException
		 */
		public static function getSenzuContent(int $siteID): array{
			$logger = MonologWrapper::getLogger();
			$uuid = Setting::getSettingValue(Settings::BUILD_UUID->value);
			$apiKey = Setting::getSettingValue(Settings::UPLIFT_CONTROL_PANEL_API_KEY->value);
			$panelHost = System::getUpliftControlPanelHost();
			$endpoint = sprintf("%s/uplift/build/senzu-importer/get-content", $panelHost);
			$client = new Client();

			$logger->info("Fetching Senzu content from control panel.");

			try {
				$response = $client->request(
					method: "GET",
					uri: $endpoint,
					options: [
						RequestOptions::QUERY => [
							"site-id" => $siteID,
						],
						RequestOptions::HEADERS => [
							"Uplift-UUID" => $uuid,
							"Uplift-API-Key" => $apiKey,
						],
					],
				);
			}catch(GuzzleException $e){
				if ($e instanceof RequestException) {
					$logger->error("Error when trying to get Senzu content. Exception message: " . $e->getResponse()->getBody()->getContents());
				}else{
					$logger->error("Error when trying to get Senzu content. Exception message: " . $e->getMessage());
				}
				throw $e;
			}

			$jsonData = $response->getBody()->getContents();

			/** @var array{contentData: array, status: int, error: ?string} | null $responseData */
			$responseData = json_decode($jsonData, true);

			if ($responseData !== null){
				$status = $responseData['status'];
				if ($status === 1) {
					$logger->info("Successfully got Senzu content from control panel API.");
					return $responseData['contentData'];
				}else{
					$logger->error("Got formatted error from Senzu control panel API endpoint.");
					throw new ControlPanelAPIException($responseData['error']);
				}
			}else{
				$logger->critical("Fatal error from API endpoint. Data received was: " . $jsonData);
				throw new ControlPanelAPIException("Invalid JSON received from Uplift control panel. Data received was: " . $jsonData);
			}
		}

		public static function changeSenzuSiteStatus(int $siteID, int $newSenzuSiteStatusID): void{
			$logger = MonologWrapper::getLogger();
			$uuid = Setting::getSettingValue(Settings::BUILD_UUID->value);
			$apiKey = Setting::getSettingValue(Settings::UPLIFT_CONTROL_PANEL_API_KEY->value);
			$panelHost = System::getUpliftControlPanelHost();
			$endpoint = sprintf("%s/uplift/build/senzu/site/status", $panelHost);
			$client = new Client();

			$logger->info("Sending request to Uplift Master Control Panel to change Senzu site ID status.");

			try {
				$response = $client->request(
					method: "PATCH",
					uri: $endpoint,
					options: [
						RequestOptions::HEADERS => [
							"Uplift-UUID" => $uuid,
							"Uplift-API-Key" => $apiKey,
						],
						RequestOptions::FORM_PARAMS => [
							"site-id" => $siteID,
							"status" => $newSenzuSiteStatusID,
						],
					],
				);
			} catch (GuzzleException $e) {
				$logger->error("Received error from Uplift Master Control panel: " . $e->getMessage());
			}
		}

		/**
		 * @throws IncompatiblePageType
		 * @throws NoPrimaryKey
		 * @throws PageLayoutDoesntExist
		 * @throws PageLayoutIsBlank
		 * @throws PageNameExists
		 * @throws PageNameExistsWithSamePageType
		 * @throws PageNameIsBlank
		 * @throws PageRouteInUse
		 * @throws PageRouteIsBlank
		 * @throws PublicationStatusDoesntExist
		 */
		public static function buildPageWithSenzuContent(
			string $pageName,
			string $pageType,
			string $pageBody,
			string $pageRoute,
			string $metaTitle,
			string $metaDescription,
			string $featuredImageURI,
			string $featuredImageThumbURI,
			array $breadcrumbs,
			array $data,
		): Page{
			$newPage = PageEditorService::createPage(
				pageName: $pageName,
				pageType:$pageType,
				overwriteExistingPage: true,
			);

			// Try and discern the page layout to set for this page
			$layouts = Layouts::getAvailableLayouts();
			$chosenLayout = "General";

			foreach ($layouts as $layout) {
				if (strtolower($layout->fileNameWithoutExtension) === strtolower($pageType) && $pageRoute !== "/") {
					$chosenLayout = $layout->fileNameWithoutExtension;
					break;
				}else{
					// Check if this layout is named "Home" and the page route is "/"
					if (strtolower($layout->fileNameWithoutExtension) === "home" && $pageRoute === "/"){
						$chosenLayout = $layout->fileNameWithoutExtension;
						break;
					}
				}
			}

			$headContents = str_replace("\t", "", trim(<<<HTML
				<title>{$metaTitle}</title>
				<meta name="description" property="og:description" content="{$metaDescription}">
			HTML));

			PageEditorService::saveBreadcrumbs(
				page: $newPage,
				breadcrumbs: $breadcrumbs,
			);

			if ($pageType === PageType::General->name) {
				PageEditorService::saveGeneralPage(
					page: $newPage,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $chosenLayout,
					pageBody: $pageBody,
					pageHead: $headContents,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
				);

				$newPage->saveToDefaultContentSectionOrFirstSection($pageBody);
				PageArchivesService::archiveGeneralPage($newPage);
			}elseif ($pageType === PageType::Service->name){
				PageEditorService::saveServicePage(
					page: $newPage,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $chosenLayout,
					pageBody: $pageBody,
					pageHead: $headContents,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
					featuredImageURI: $featuredImageURI,
					featuredImageThumbURI: $featuredImageThumbURI,
				);

				$newPage->saveToDefaultContentSectionOrFirstSection($pageBody);
				PageArchivesService::archiveServicePage($newPage);
			}elseif ($pageType === PageType::City->name){
				PageEditorService::saveCityPage(
					page: $newPage,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $chosenLayout,
					pageBody: $pageBody,
					pageHead: $headContents,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
					cityName: $data[PageDatas::CITY_NAME->name] ?? "",
					stateName: $data[PageDatas::STATE_NAME->name] ?? "",
					stateNameShorthand: $data[PageDatas::STATE_NAME_SHORTHAND->name] ?? "",
					officialCityURL: $data[PageDatas::OFFICIAL_CITY_URL->name] ?? "",
					featuredImageURI: $featuredImageURI,
					featuredImageThumbURI: $featuredImageThumbURI,
				);

				$newPage->saveToDefaultContentSectionOrFirstSection($pageBody);
				PageArchivesService::archiveCityPage($newPage);
			}elseif ($pageType === PageType::Blog->name){
				$categoryIDs = $data[PageDatas::ARTICLE_CATEGORY->name] ?? [];
				if (!is_array($categoryIDs)){
					$categoryIDs = [$categoryIDs];
				}

				PageEditorService::saveBlogPage(
					page: $newPage,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $chosenLayout,
					pageBody: $pageBody,
					pageHead: $headContents,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
					featuredImageURI: $featuredImageURI,
					featuredImageThumbURI: $featuredImageThumbURI,
					articleCategoryIDs: $categoryIDs,
				);

				$newPage->saveToDefaultContentSectionOrFirstSection($pageBody);
				PageArchivesService::archiveBlogPage($newPage);
			}elseif ($pageType === PageType::Project->name){
				$projectTagIDs = $data[PageDatas::PROJECT_TAG->name] ?? [];
				if (!is_array($projectTagIDs)){
					$projectTagIDs = [$projectTagIDs];
				}

				PageEditorService::saveProjectPage(
					page: $newPage,
					pageName: $pageName,
					pageRoute: $pageRoute,
					pageRouteIsRegex: false,
					pageLayout: $chosenLayout,
					pageBody: $pageBody,
					pageHead: $headContents,
					excludedFromSitemap: false,
					excludedFromSchema: false,
					publicationStatus: PublicationStatus::Published->value,
					publicationTimestamp: time(),
					featuredImageURI: $featuredImageURI,
					featuredImageThumbURI: $featuredImageThumbURI,
					cityName: $data[PageDatas::CITY_NAME->name] ?? "",
					stateName: $data[PageDatas::STATE_NAME->name] ?? "",
					stateNameShorthand: $data[PageDatas::STATE_NAME_SHORTHAND->name] ?? "",
					brandsProducts: $data[PageDatas::PROJECT_BRANDS_PRODUCTS->name] ?? "",
					customerDidLeaveReview: $data[PageDatas::CUSTOMER_DID_LEAVE_REVIEW->name] ?? "0",
					customerReviewFirstName: $data[PageDatas::CUSTOMER_REVIEW_FIRST_NAME->name] ?? "",
					customerReviewLastName: $data[PageDatas::CUSTOMER_REVIEW_LAST_NAME->name] ?? "",
					customerReviewTestimonial: $data[PageDatas::CUSTOMER_REVIEW_TESTIMONIAL->name] ?? "",
					projectTagIDs: $projectTagIDs,
				);

				$newPage->saveToDefaultContentSectionOrFirstSection($pageBody);
				PageArchivesService::archiveProjectPage($newPage);
			}

			return $newPage;
		}
	}