<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";
	require_once __DIR__ . "/ShortcodeViewProcessor.php";

	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\ContentHelper;
	use System\Themes;

	/**
	 * Processor class for the get-articles shortcode
	 */
	class GetArticles extends ShortcodeViewProcessor {

		protected static string $defaultShortcodeViewBaseFileName = "articles-reel.php";

		protected static array $supportedTemplateTags = [
			"{{ COLUMNS }}",
			"{{ PER_PAGE_LIMIT }}",
			"{{ begin PageItem }}",
			"{{ end PageItem }}",
			"{{ ARTICLE_TITLE }}",
			"{{ FEATURED_IMAGE }}",
			"{{ FEATURED_IMAGE_NOTHUMB }}",
			"{{ ARTICLE_PREVIEW }}",
			"{{ ARTICLE_URI }}",
			"{{ CREATION_DATE }}",
		];

		protected static array $supportedAttributes = [
			"articles-per-page",
			"columns",
			"excluded-categories",
			"included-categories",
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public static function getDefaultShortcodeViewBaseFileName(): string{
			return self::$defaultShortcodeViewBaseFileName;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws MissingRequiredAttribute
		 * @throws InvalidAttributeValue
		 * @throws Exception
		 */
		public function runProcessor(): string{

			// Required attribute
			$articlesPerPageAttribute = $this->shortcode->getAttribute("articles-per-page");
			$articlesPerPage = 2;
			if ($articlesPerPageAttribute === null){
				throw new MissingRequiredAttribute(
					sprintf(
						"Attribute 'articles-per-page' is required in %s",
						ShortcodeTypes::GET_ARTICLES->value,
					)
				);
			}else{
				$articlesPerPage = (int) $articlesPerPageAttribute->value;
			}

			// Handle columns + validation
			$columnsAttribute = $this->shortcode->getAttribute("columns");
			$columns = null;
			if ($columnsAttribute === null){
				$columns = 1;
			}else{
				$columns = (int) $columnsAttribute->value;
				if ($columns === 0 || $columns < 0){
					throw new InvalidAttributeValue("The value of 'columns' attribute must be a valid, positive integer.");
				}
			}

			// Exclusion and inclusion article category IDs
			$includedArticleCategoryIDsAttribute = $this->shortcode->getAttribute("included-categories");
			$includedCategoryIDs = [];
			if ($includedArticleCategoryIDsAttribute !== null){
				// JSON string, then parse it
				$jsonString = $includedArticleCategoryIDsAttribute->value;
				if (!empty($jsonString)) {
					$jsonIncludedIDs = json_decode($includedArticleCategoryIDsAttribute->value, true);
					if ($jsonIncludedIDs !== null) {
						$includedCategoryIDs = $jsonIncludedIDs;
					}else{
						throw new InvalidAttributeValue(
							sprintf(
								"The JSON value for 'included-categories' was invalid. The error received was: %s",
								json_last_error_msg(),
							)
						);
					}
				}
			}

			$excludedArticleCategoryIDsAttribute = $this->shortcode->getAttribute("excluded-categories");
			$excludedCategoryIDs = [];
			if ($excludedArticleCategoryIDsAttribute !== null){
				// JSON string, then parse it
				$jsonString = $excludedArticleCategoryIDsAttribute->value;
				if (!empty($jsonString)) {
					$jsonExcludedIDs = json_decode($excludedArticleCategoryIDsAttribute->value, true);
					if ($jsonExcludedIDs !== null) {
						$excludedCategoryIDs = $jsonExcludedIDs;
					}else{
						throw new InvalidAttributeValue(
							sprintf(
								"The JSON value for 'excluded-categories' was invalid. The error received was: %s",
								json_last_error_msg(),
							)
						);
					}
				}
			}

			// Shortcode folder
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();

			// Get the view file and then its contents
			$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, self::$defaultShortcodeViewBaseFileName);
			$viewFileNormalizedPath = realpath($viewFile);
			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			// Replace the template tags
			$viewFileContents = str_replace(
				search: "{{ COLUMNS }}",
				replace: $columns,
				subject: $viewFileContents,
			);

			$viewFileContents = str_replace(
				search: "{{ PER_PAGE_LIMIT }}",
				replace: $articlesPerPage,
				subject: $viewFileContents,
			);

			// Replace the template tag HTML wrapper with the necessary articles to show

			$viewFileContents = preg_replace_callback(
				pattern:"/{{ begin PageItem }}(.+?){{ end PageItem }}/ism",
				callback: function($matches) use ($includedCategoryIDs, $excludedCategoryIDs, $articlesPerPage){
					$template = $matches[1];
					$finalStringToRender = "";
					// To be optimized, the following algorithm has been devised:
					/**
					 * If the included category IDs array is not empty:
					 * 	Fetch all PageDatas that are of ARTICLE_CATEGORY, then put them into an array
					 * 	where the key is "pageID" and value is an array of article category IDs.
					 * 	Put that array into an encompassing array of "fetchedPageDatas".
					 * 	Then, if the excluded category IDs array is not empty,
					 * 	iterate over all elements of fetchedPageDatas and remove
					 * 	any keys where any of the excluded category IDs are found in the articleCategories key.
					 * Else, if the included category IDs array is empty then a heavy-query will have to be performed.
					 * 	If excluded category IDs is not empty:
					 * 		1) Fetch _all_ blog pages that are published, and the publication timestamp is either today or prior,
					 * 		sorted by first their publication timestamp DESC and then creation timestamp DESC.
					 * 		2) Now, if excluded category IDs are not empty: Iterate over each page one by and fetch the
					 * 		PageDatas of type ARTICLE_CATEGORY WHERE the value is in the excluded category IDs. If no
					 * 		match, then add it to the final fetched article pages.
					 * If neither included category IDs nor excluded category IDs, simply fetch all articles that are
					 * marked as published, sort by publication timestamp DESC, creation timestamp DESC, and then limit
					 * it by the number of articles requested by the shortcode.
					 */

					/** @var Page[] $pagesToRender */
					$pagesToRender = [];

					// Handle where included category IDs is not empty
					if (!empty($includedCategoryIDs)){

						/** @var array[] $pageArticleCategories */
						$pageArticleCategories = [];

						/** @var PageData[] $pageDatas */
						$pageDatas = PageData::query(
							columnQuery: (new ColumnQuery())
								->where("name","=",PageDatas::ARTICLE_CATEGORY->name)
						);

						// Build an array of article categories identified by the page ID
						foreach($pageDatas as $pageData){
							if(in_array((int) $pageData->value, $includedCategoryIDs)){
								if (array_key_exists(key: $pageData->pageID, array: $pageArticleCategories)){
									$pageArticleCategories[$pageData->pageID][] = (int) $pageData->value;
								}else{
									$pageArticleCategories[$pageData->pageID] = [
										(int) $pageData->value,
									];
								}
							}
						}

						// Filter out pageIDs that have an excluded article category ID - if any
						if (!empty($excludedCategoryIDs)) {
							foreach ($pageArticleCategories as $pageID => $articleCategoryIDsForPage) {
								// If this array is empty, then the article isn't excluded,
								// else if the array is not empty then the article contains an
								// excluded category.
								if (!empty(array_intersect($articleCategoryIDsForPage, $excludedCategoryIDs))) {
									// Unset it
									unset($pageArticleCategories[$pageID]);
								}
							}
						}

						// If this is empty, then the query would throw an error (MySQL syntax: 'IN ()' is invalid)
						if (!empty($pageArticleCategories)) {
							// Fetch the pages now
							/** @var Page[] $pagesToRender */
							$pagesToRender = Page::query(
								columnQuery: (new ColumnQuery())
									->where("pageType", "=", PageType::Blog->name)
									->and()
									->where("publication_status", "=", PublicationStatus::Published->value)
									->and()
									->where("publication_timestamp", "<=", time())
									->and()
									->where("id", "IN", sprintf(
										"(%s)",
										implode(
											",",
											array_keys($pageArticleCategories) // Keys are page IDs
										)
									)),
								resultOrder: (new ResultOrder())
									->by("publication_timestamp", "DESC")
									->by("creationTime", "DESC"),
							);
						}
					}else{
						// Included category IDs is empty
						// Are there excluded categories?
						if (!empty($excludedCategoryIDs)){

							/** @var array[] $pageArticleCategories */
							$pageArticleCategories = [];

							/** @var PageData[] $pageDatas */
							$pageDatas = PageData::query(
								columnQuery: (new ColumnQuery())
									->where("name","=",PageDatas::ARTICLE_CATEGORY->name)
							);

							// Build an array of article categories identified by the page ID
							foreach($pageDatas as $pageData){
								if (array_key_exists(key: $pageData->pageID, array: $pageArticleCategories)){
									$pageArticleCategories[$pageData->pageID][] = (int) $pageData->value;
								}else{
									$pageArticleCategories[$pageData->pageID] = [
										(int) $pageData->value,
									];
								}
							}

							foreach ($pageArticleCategories as $pageID => $articleCategoryIDsForPage) {
								// If this array is empty, then the article isn't excluded,
								// else if the array is not empty then the article contains an
								// excluded category.
								if (!empty(array_intersect($articleCategoryIDsForPage, $excludedCategoryIDs))) {
									// Unset it
									unset($pageArticleCategories[$pageID]);
								}
							}

							// If this is empty, then the query would throw an error (MySQL syntax: 'IN ()' is invalid)
							if (!empty($pageArticleCategories)) {
								// Fetch the pages now
								/** @var Page[] $pagesToRender */
								$pagesToRender = Page::query(
									columnQuery: (new ColumnQuery())
										->where("pageType", "=", PageType::Blog->name)
										->and()
										->where("publication_status", "=", PublicationStatus::Published->value)
										->and()
										->where("publication_timestamp", "<=", time())
										->and()
										->where("id", "IN", sprintf(
											"(%s)",
											implode(
												",",
												array_keys($pageArticleCategories) // Keys are page IDs
											)
										)),
									resultOrder: (new ResultOrder())
										->by("publication_timestamp", "DESC")
										->by("creationTime", "DESC"),
								);
							}
						}else{
							// At this point there are no included category IDs and no excluded IDs.
							// This is a simple, limited fetch query of articles that are published
							/** @var Page[] $pagesToRender */
							$pagesToRender = Page::query(
								columnQuery: (new ColumnQuery())
									->where("pageType", "=", PageType::Blog->name)
									->and()
									->where("publication_status", "=", PublicationStatus::Published->value)
									->and()
									->where("publication_timestamp","<=", time()),
								resultOrder:(new ResultOrder())
									->by("publication_timestamp","DESC")
									->by("creationTime","DESC"),
							);
						}
					}

					// Render the articles into the shortcode view now
					foreach($pagesToRender as $page){
						$currentPage = Page::getCurrentPage();

						// Make sure to ignore this page if it is in the loop
						if ($currentPage->id === $page->id){
							continue;
						}

						// Make a copy of the string
						$thisPageRenderString = $template;

						// Make all template tag replacements
						$thisPageRenderString = str_replace(
							search:"{{ ARTICLE_TITLE }}",
							replace:$page->pageName,
							subject:$thisPageRenderString,
						);

						$featuredImageDataArray = $page->getPageDatas(PageDatas::FEATURED_IMAGE);
						$featuredImageThumbDataArray = $page->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB);

						if (!empty($featuredImageThumbDataArray)) {
							$thisPageRenderString = str_replace(
								search: "{{ FEATURED_IMAGE }}", // This actually is the thumb template tag. I know, dumb
								replace: $featuredImageThumbDataArray[0]->value,
								subject: $thisPageRenderString,
							);
						}

						if (!empty($featuredImageDataArray)) {
							$thisPageRenderString = str_replace(
								search: "{{ FEATURED_IMAGE_NOTHUMB }}",
								replace: $featuredImageDataArray[0]->value,
								subject: $thisPageRenderString,
							);
						}

						// Get a preview text for this article's default/first content section, or page body if no sections
						$defaultOrFirstContentSection = $page->getDefaultContentSectionOrFirstSection();
						$previewText = ContentHelper::makePreviewFromBody(
							pageBody: $defaultOrFirstContentSection?->content ?? $page->pageBody,
						);

						$thisPageRenderString = str_replace(
							search: "{{ ARTICLE_PREVIEW }}",
							replace: $previewText . "[&hellip;]",
							subject: $thisPageRenderString,
						);

						$thisPageRenderString = str_replace(
							search: "{{ ARTICLE_URI }}",
							replace: $page->pageRoute,
							subject: $thisPageRenderString,
						);

						$thisPageRenderString = str_replace(
							search: "{{ CREATION_DATE }}",
							replace: date(
								format: "F j, Y",
								timestamp: $page->creationTime
							),
							subject: $thisPageRenderString,
						);

						$finalStringToRender .= $thisPageRenderString;
					}

					return $finalStringToRender;
				},
				subject:$viewFileContents
			);

			return $viewFileContents;
		}
	}