<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";
	require_once __DIR__ . "/ShortcodeViewProcessor.php";

	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use Reviews\Review;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\NoPageCurrentlyRendered;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeParser;
	use ShortcodeParser\ShortcodeTypes;
	use System\ContentHelper;
	use System\HttpHelper;
	use System\Themes;

	/**
	 * Processor class for the breadcrumbs shortcode
	 */
	class GetBreadcrumbs extends ShortcodeViewProcessor {

		protected static string $defaultShortcodeViewBaseFileName = "breadcrumbs.php";

		protected static array $supportedTemplateTags = [
			"{{ begin BreadcrumbItem }}",
			"{{ end BreadcrumbItem }}",
			"{{ LABEL }}",
			"{{ PAGE_URI }}",
		];

		protected static array $supportedAttributes = [
			"schema",
			"shortcode-file",
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public static function getDefaultShortcodeViewBaseFileName(): string{
			return self::$defaultShortcodeViewBaseFileName;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @return string
		 * @throws NoPageCurrentlyRendered
		 * @throws ShortcodeViewFileNotFound
		 * @throws Exception
		 */
		public function runProcessor(): string{
			$schemaAttribute = $this->shortcode->getAttribute("schema");
			$hasSchema = true; // Defaults to true
			if ($schemaAttribute !== null){
				if ($schemaAttribute->value === "no" || $schemaAttribute->value === "false"){
					$hasSchema = false;
				}
			}

			// Shortcode folder
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();

			// Is this using a custom shortcode view file?
			$shortcodeViewFileAttribute = $this->shortcode->getAttribute("shortcode-file");
			if ($shortcodeViewFileAttribute !== null){
				// Fetch the file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, $shortcodeViewFileAttribute->value);
			}else{
				// Use the default file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, self::$defaultShortcodeViewBaseFileName);
			}

			// Get the view file and then its contents
			$viewFileNormalizedPath = realpath($viewFile);
			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			$shortcodeParser = new ShortcodeParser();
			// Replace the template tag HTML wrapper with the necessary articles to show
			$viewFileContents = preg_replace_callback(
				pattern:"/{{ begin BreadcrumbItem }}(.+?){{ end BreadcrumbItem }}/ism",
				callback: function($matches) use ($hasSchema, $shortcodeParser){
					$template = $matches[1];
					$finalStringToRender = "";

					// Fetch current page
					$currentPage = Page::getCurrentPage();

					if ($currentPage === null){
						throw new NoPageCurrentlyRendered("There is no current Uplift CMS page being rendered in the context that getCurrentPage() was called.");
					}

					if ($hasSchema){
						$schema = [
							"@context"=>"http://schema.org",
							"@type"=>"BreadcrumbList",
							"itemListElement"=>[],
						];
					}

					foreach($currentPage->getBreadcrumbs() as $crumb){
						$thisCrumbRenderedString = $template;

						$parsedLabel = $crumb->label;

						// Historically breadcrumb labels also get parsed for shortcodes

						$parsedLabel = $shortcodeParser->parse($parsedLabel);
						$sanitizedLabel = strip_tags($parsedLabel);

						$thisCrumbRenderedString = str_replace(
							search:"{{ LABEL }}",
							replace:$parsedLabel,
							subject:$thisCrumbRenderedString,
						);

						$thisCrumbRenderedString = str_replace(
							search:"{{ PAGE_URI }}",
							replace:$crumb->uri,
							subject:$thisCrumbRenderedString,
						);

						if ($hasSchema){
							$schema['itemListElement'][] = [
								"@type"=>"ListItem",
								"position"=>$crumb->position,
								"item"=>[
									"@id"=>HttpHelper::getWebsiteBaseURL() . $crumb->uri,
									"name"=>$sanitizedLabel
								]
							];
						}

						$finalStringToRender .= $thisCrumbRenderedString;
					}

					if ($hasSchema){
						$schemaScript = sprintf(
							"<script type=\"application/ld+json\">%s</script>\n",
							json_encode($schema)
						);
						return $schemaScript .= $finalStringToRender;
					}else{
						return $finalStringToRender;
					}
				},
				subject:$viewFileContents
			);

			return $viewFileContents;
		}
	}