<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";

	use Page\Page;
	use Page\PageDatas;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\NoPageCurrentlyRendered;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;

	/**
	 * Processor class for the city-name shortcode
	 */
	class GetCityName extends Processor {

		protected static array $supportedAttributes = [
			"map-safe",
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws NoPageCurrentlyRendered|InvalidAttributeValue
		 */
		public function runProcessor(): string{
			$mapSafeAttribute = $this->shortcode->getAttribute("map-safe");
			$shouldFormatToMapSafe = false;

			if ($mapSafeAttribute !== null){
				if (isset($mapSafeAttribute->value)){
					if ($mapSafeAttribute->value === "false") {
						$shouldFormatToMapSafe = false;
					} elseif ($mapSafeAttribute->value === "true") {
						$shouldFormatToMapSafe = true;
					} else {
						throw new InvalidAttributeValue("map-safe attribute value should be either the string 'false' or 'true' and not " . $mapSafeAttribute->value);
					}
				}else{
					// The attribute is present with no value - assume true
					$shouldFormatToMapSafe = true;
				}
			}

			// Fetch the current page, if any
			$currentPage = Page::getCurrentPage();
			if ($currentPage !== null){
				// Attempt to get the city name
				$cityNameDatas = $currentPage->getPageDatas(PageDatas::CITY_NAME);
				if (empty($cityNameDatas)){
					// No city name for this page
					// Return blank
					return "";
				}else{
					// Get the first result
					$cityNameData = $cityNameDatas[0];
					$cityName = $cityNameData->value;

					// Does it need to be map-safe?
					if ($shouldFormatToMapSafe){
						$cityName = preg_replace(
							pattern:"@\s@",
							replacement: "+",
							subject: $cityName,
						);
					}

					return $cityName;
				}
			}else{
				throw new NoPageCurrentlyRendered(
					sprintf(
						"Failed to run %s shortcode. There is no current Uplift CMS page being rendered.",
						ShortcodeTypes::GET_CITY_NAME->value,
					)
				);
			}
		}
	}
