<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";

	use Page\Page;
	use Page\PageDatas;
	use Settings\Setting;
	use Settings\Settings;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\MissingSettingValueForShortcode;
	use ShortcodeParser\Processors\Exceptions\NoPageCurrentlyRendered;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\PhoneNumberUtility;

	/**
	 * Processor class for the phone-number shortcode
	 */
	class GetCompanyPhoneNumber extends Processor {

		protected static array $supportedAttributes = [
			"which",
			"format",
			"pure-number"
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws MissingRequiredAttribute
		 * @throws InvalidAttributeValue
		 * @throws MissingSettingValueForShortcode
		 */
		public function runProcessor(): string{
			$phoneNumbersJSON = Setting::getSettingValue(Settings::COMPANY_PHONE_NUMBERS->value);
			if ($phoneNumbersJSON !== null) {
				$phoneNumbers = json_decode($phoneNumbersJSON, true);
				if (is_array($phoneNumbers) && !empty($phoneNumbers)) {
					$isPureNumber = false;

					$whichIndexAttribute = $this->shortcode->getAttribute("which");
					if ($whichIndexAttribute === null) {
						throw new MissingRequiredAttribute(
							sprintf(
								"Missing required 'which' attribute on shortcode %s",
								ShortcodeTypes::PHONE_NUMBER->value,
							)
						);
					} else {

						// Catch an error that happens when a writer, or developer, would use the following format
						// {{ phone-number which 1 }}
						if (!isset($whichIndexAttribute->value)){
							throw new MissingRequiredAttribute(
								sprintf(
									"Malformed 'which' attribute on shortcode %s - missing the value (is it missing an equal sign?).",
									ShortcodeTypes::PHONE_NUMBER->value,
								)
							);
						}

						$whichIndex = (int)$whichIndexAttribute->value;
					}

					// Only care if 'pure-number' is there and not equal to "false" or "no"
					$pureNumberAttribute = $this->shortcode->getAttribute("pure-number");
					if ($pureNumberAttribute !== null) {
						if ($pureNumberAttribute->value !== "false" && $pureNumberAttribute->value !== "no") {
							$isPureNumber = true;
						}
					}

					if ($whichIndex <= 0) {
						throw new InvalidAttributeValue(
							"The value for 'which' must be greater than or equal to 1."
						);
					}

					// Get the actual array index. Which's index is 1-based. We need 0 based
					$arrayIndex = $whichIndex - 1;
					$phoneNumber = $phoneNumbers[$arrayIndex];

					// A pure number will convert any letters to their phone-digit
					if ($isPureNumber) {
						$phoneNumber = PhoneNumberUtility::getPurePhoneNumber($phoneNumber);
					}

					$formatAttribute = $this->shortcode->getAttribute("format");
					if ($formatAttribute !== null) {
						$numberFormat = $formatAttribute->value;

						$numberOfDigitsInPhoneNumber = preg_match_all("/[0-9]/", $phoneNumber, $phoneNumberDigitsMatch);
						$numberOfCharactersForFormatting = substr_count($numberFormat, 'x');

						if ($numberOfDigitsInPhoneNumber !== $numberOfCharactersForFormatting) {
							// There is not an equal amount of phone number digit characters and format characters
							throw new InvalidAttributeValue(
								sprintf(
									"The value for the 'format' attribute is malformed. The number of numerical digits in the company phone number is not equal to the amount of format characters in the 'format' attribute (x's). They must be equal. You provided %d format characters but the phone number requires there to be %d. Company phone number being used is: %s | You provided the format string %s",
									$numberOfCharactersForFormatting,
									$numberOfDigitsInPhoneNumber,
									$phoneNumber,
									$numberFormat,
								)
							);
						}

						foreach ($phoneNumberDigitsMatch[0] as $char) {
							$from = '/' . preg_quote('x', '/') . '/';
							$phoneNumber = preg_replace(
								pattern: $from,
								replacement: $char,
								subject: $phoneNumber,
								limit: 1
							);
						}
					}

					return $phoneNumber;
				}
			}

			throw new MissingSettingValueForShortcode(
				sprintf(
					"There is no setting defined in the system for %s - which is required for the shortcode %s",
					Settings::COMPANY_PHONE_NUMBERS->value,
					ShortcodeTypes::PHONE_NUMBER->value,
				)
			);
		}
	}
