<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";
	require_once __DIR__ . "/ShortcodeViewProcessor.php";

	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\ContentHelper;
	use System\Themes;

	/**
	 * Processor class for the get-recent-articles shortcode
	 */
	class GetRecentArticles extends ShortcodeViewProcessor {

		protected static string $defaultShortcodeViewBaseFileName = "recent-articles.php";

		protected static array $supportedTemplateTags = [
			"{{ COLUMNS }}",
			"{{ NUM_ITEMS }}",
			"{{ begin RecentArticleItem }}",
			"{{ ARTICLE_TITLE }}",
			"{{ ARTICLE_PREVIEW }}",
			"{{ COLUMN_CLASS }}",
			"{{ CURRENT_INDEX }}",
			"{{ ARTICLE_THUMB }}",
			"{{ FEATURED_IMAGE }}",
			"{{ FEATURED_IMAGE_NOTHUMB }}",
			"{{ ARTICLE_URI }}",
			"{{ PAGE_URI }}",
            "{{ PUBLISHED_DATE }}",
		];

		protected static array $supportedAttributes = [
			"num-articles",
			"columns",
			"shortcode-file",
			"included-article-categories",
			"preview-max-word-count",
			"placeholder-data"
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public static function getDefaultShortcodeViewBaseFileName(): string{
			return self::$defaultShortcodeViewBaseFileName;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws MissingRequiredAttribute
		 * @throws InvalidAttributeValue
		 * @throws Exception
		 */
		public function runProcessor(): string{

			// Required attribute
			$numArticlesAttribute = $this->shortcode->getAttribute("num-articles");
			if ($numArticlesAttribute === null){
				throw new MissingRequiredAttribute(
					sprintf(
						"Attribute 'num-articles' is required in %s",
						ShortcodeTypes::GET_RECENT_ARTICLES->value,
					)
				);
			}else{
				$numArticles = (int) $numArticlesAttribute->value;
			}

			// Handle columns + validation
			$columnsAttribute = $this->shortcode->getAttribute("columns");
			if ($columnsAttribute === null){
				$columns = 1;
			}else{
				$columns = (int) $columnsAttribute->value;
				if ($columns === 0 || $columns < 0){
					throw new InvalidAttributeValue("The value of 'columns' attribute must be a valid, positive integer.");
				}
			}

			// Shortcode folder
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();

			// Is this using a custom shortcode view file?
			$shortcodeViewFileAttribute = $this->shortcode->getAttribute("shortcode-file");
			if ($shortcodeViewFileAttribute !== null){
				// Fetch the file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, $shortcodeViewFileAttribute->value);
			}else{
				// Use the default file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, self::$defaultShortcodeViewBaseFileName);
			}

			// Get the view file and then its contents
			$viewFileNormalizedPath = realpath($viewFile);
			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			// Replace the template tags
			$viewFileContents = str_replace(
				search: "{{ COLUMNS }}",
				replace: $columns,
				subject: $viewFileContents,
			);

			// TODO
			// Replace this with something more scalable and customizable
			// Hard-coded Bootstrap column classes might limit us in the future
			$columnClassStub = "col-lg-%s"; // %s because this will be used in a string format function (sprintf)
			$classColNum = 12 / $columns;

            if($this->shortcode->getAttribute("placeholder-data")?->value === "true") {
                $viewFileContents = preg_replace_callback(
                    pattern:"/{{ begin RecentArticleItem }}(.+?){{ end RecentArticleItem }}/ism",
                    callback: function($matches) use ($numArticles, $columnClassStub, $classColNum){
                        $template = $matches[1];
                        $finalStringToRender = "";

                        for($i = 0; $i < $numArticles; $i++) {
                            // Make a copy of the string
                            $thisPageRenderString = $template;

                            // Make all template tag replacements
                            $thisPageRenderString = str_replace(
                                search:"{{ ARTICLE_TITLE }}",
                                replace:"Placeholder Article Title " . $i + 1,
                                subject:$thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search:"{{ CURRENT_INDEX }}",
                                replace:$i + 1, // 1-based count, instead of 0-based count
                                subject:$thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search:"{{ COLUMN_CLASS }}",
                                replace:sprintf($columnClassStub, $classColNum),
                                subject:$thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search:"{{ FEATURED_IMAGE }}", // This actually is the thumb template tag. I know, dumb
                                replace:"/uplift-data/images/articles/thumbs/article-" . $i + 1 . ".jpg",
                                subject:$thisPageRenderString,
                            );

                            // Old name for it, sorry
                            $thisPageRenderString = str_replace(
                                search:"{{ ARTICLE_THUMB }}",
                                replace:"/uplift-data/images/articles/thumbs/article-" . $i + 1 . ".jpg",
                                subject:$thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search:"{{ FEATURED_IMAGE_NOTHUMB }}",
                                replace:"/uplift-data/images/articles/article-" . $i + 1 . ".jpg",
                                subject:$thisPageRenderString,
                            );

                            $previewMaxWordCountAttribute = $this->shortcode->getAttribute("preview-max-word-count");
                            $previewMaxWordCount = 45;

                            if ($previewMaxWordCountAttribute) {
                                $previewMaxWordCount = (int) $previewMaxWordCountAttribute->value;
                            }

                            $previewText = ContentHelper::makePreviewFromBody(
                                pageBody: "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Nunc sit amet blandit arcu. Proin placerat, quam ac tincidunt varius, mi odio molestie massa, sit amet laoreet erat enim et libero. In rutrum mi dui, id condimentum quam malesuada quis. Sed elementum, diam in vulputate fringilla, eros augue convallis augue, non pharetra neque justo sit amet risus. Mauris porttitor ipsum a dui molestie tristique ac id quam. Nam volutpat, urna eu rutrum pharetra, diam elit gravida leo, ac gravida lorem neque at velit. Phasellus hendrerit non tortor sodales hendrerit.",
                                maxWordCount: $previewMaxWordCount
                            );

                            $thisPageRenderString = str_replace(
                                search: "{{ ARTICLE_PREVIEW }}",
                                replace: $previewText . "[&hellip;]",
                                subject: $thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search:"{{ ARTICLE_URI }}",
                                replace:"#link",
                                subject:$thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search:"{{ PAGE_URI }}",
                                replace:"#link",
                                subject:$thisPageRenderString,
                            );

                            $publishedDate = date("F j, Y");

                            $thisPageRenderString = str_replace(
                                search: "{{ PUBLISHED_DATE }}",
                                replace: $publishedDate,
                                subject: $thisPageRenderString,
                            );

                            $finalStringToRender .= $thisPageRenderString;
                        }

                        return $finalStringToRender;
                    },
                    subject:$viewFileContents
                );

                return $viewFileContents;
            } else {
                // Inclusion article category IDs
                $includedArticleCategoryIDsAttribute = $this->shortcode->getAttribute("included-article-categories");
                $includedCategoryIDs = [];
                if ($includedArticleCategoryIDsAttribute !== null){
                    // JSON string, then parse it
                    $jsonString = $includedArticleCategoryIDsAttribute->value;
                    if (!empty($jsonString)) {
                        $jsonIncludedIDs = json_decode($includedArticleCategoryIDsAttribute->value, true);
                        if ($jsonIncludedIDs !== null) {
                            $includedCategoryIDs = $jsonIncludedIDs;
                        }else{
                            throw new InvalidAttributeValue(
                                sprintf(
                                    "The JSON value for 'included-article-categories' was invalid. The error received was: %s",
                                    json_last_error_msg(),
                                )
                            );
                        }
                    }
                }

                // Replace the template tag HTML wrapper with the necessary articles to show
                $viewFileContents = preg_replace_callback(
                    pattern:"/{{ begin RecentArticleItem }}(.+?){{ end RecentArticleItem }}/ism",
                    callback: function($matches) use ($includedCategoryIDs, $numArticles, $columnClassStub, $classColNum){
                        $template = $matches[1];
                        $finalStringToRender = "";

                        /** @var Page[] $pagesToRender */
                        $pagesToRender = [];

                        // Handle when specific articles categories are desired
                        if (!empty($includedCategoryIDs)){
                            /** @var array[] $pageArticleCategories */
                            $pageArticleCategories = [];
                            /** @var PageData[] $pageDatas */
                            $pageDatas = PageData::query(
                                columnQuery: (new ColumnQuery())
                                    ->where("name","=",PageDatas::ARTICLE_CATEGORY->name)
                                    ->and()
                                    ->where("value", "IN", sprintf("(%s)", implode(",", $includedCategoryIDs))),
                            );

                            // Build an array of article categories identified by the page ID
                            foreach($pageDatas as $pageData){
                                if (array_key_exists(key: $pageData->pageID, array: $pageArticleCategories)){
                                    $pageArticleCategories[$pageData->pageID][] = (int) $pageData->value;
                                }else{
                                    $pageArticleCategories[$pageData->pageID] = [
                                        (int) $pageData->value,
                                    ];
                                }
                            }

                            // If this is empty, then the query would throw an error (MySQL syntax: 'IN ()' is invalid)
                            if (!empty($pageArticleCategories)) {
                                // Fetch the pages now
                                /** @var Page[] $pagesToRender */
                                $pagesToRender = Page::query(
                                    columnQuery: (new ColumnQuery())
                                        ->where("pageType", "=", PageType::Blog->name)
                                        ->and()
                                        ->where("publication_status", "=", PublicationStatus::Published->value)
                                        ->and()
                                        ->where("publication_timestamp", "<=", time())
                                        ->and()
                                        ->where("id", "IN", sprintf(
                                            "(%s)",
                                            implode(
                                                ",",
                                                array_keys($pageArticleCategories) // Keys are page IDs
                                            )
                                        )),
                                    resultOrder: (new ResultOrder())
                                        ->by("publication_timestamp", "DESC")
                                        ->by("creationTime", "DESC"),
                                    pager: (new Pager(pageNumber: 1, limit: $numArticles))
                                );
                            }
                        }else{
                            // Fetch just the most recent articles without any category restrictions
                            /** @var Page[] $pagesToFilter */
                            $pagesToRender = Page::query(
                                columnQuery: (new ColumnQuery())
                                    ->where("pageType", "=", PageType::Blog->name)
                                    ->and()
                                    ->where("publication_status", "=", PublicationStatus::Published->value)
                                    ->and()
                                    ->where("publication_timestamp","<=", time()),
                                resultOrder:(new ResultOrder())
                                    ->by("publication_timestamp","DESC")
                                    ->by("creationTime","DESC"),
                                pager:(new Pager(pageNumber: 1, limit: $numArticles)),
                            );
                        }

                        // Render the articles into the shortcode view now
                        foreach($pagesToRender as $index=>$page){
                            // Make a copy of the string
                            $thisPageRenderString = $template;

                            // Make all template tag replacements
                            $thisPageRenderString = str_replace(
                                search:"{{ ARTICLE_TITLE }}",
                                replace:$page->pageName,
                                subject:$thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search:"{{ CURRENT_INDEX }}",
                                replace:$index + 1, // 1-based count, instead of 0-based count
                                subject:$thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search:"{{ COLUMN_CLASS }}",
                                replace:sprintf($columnClassStub, $classColNum),
                                subject:$thisPageRenderString,
                            );

                            $featuredImageDataArray = $page->getPageDatas(PageDatas::FEATURED_IMAGE);
                            $featuredImageThumbDataArray = $page->getPageDatas(PageDatas::FEATURED_IMAGE_THUMB);

                            if (!empty($featuredImageThumbDataArray)) {
                                $thisPageRenderString = str_replace(
                                    search: "{{ FEATURED_IMAGE }}", // This actually is the thumb template tag. I know, dumb
                                    replace: $featuredImageThumbDataArray[0]->value,
                                    subject: $thisPageRenderString,
                                );

                                // Old name for it, sorry
                                $thisPageRenderString = str_replace(
                                    search: "{{ ARTICLE_THUMB }}",
                                    replace: $featuredImageThumbDataArray[0]->value,
                                    subject: $thisPageRenderString,
                                );
                            }

                            if (!empty($featuredImageDataArray)) {
                                $thisPageRenderString = str_replace(
                                    search: "{{ FEATURED_IMAGE_NOTHUMB }}",
                                    replace: $featuredImageDataArray[0]->value,
                                    subject: $thisPageRenderString,
                                );
                            }

                            // Handle preview-max-word-count
                            $previewMaxWordCountAttribute = $this->shortcode->getAttribute("preview-max-word-count");
                            $previewMaxWordCount = 45;

                            if ($previewMaxWordCountAttribute) {
                                $previewMaxWordCount = (int) $previewMaxWordCountAttribute->value;
                            }

                            // Get a preview text for this article's default/first content section, or page body if no sections
                            $defaultOrFirstContentSection = $page->getDefaultContentSectionOrFirstSection();
                            $previewText = ContentHelper::makePreviewFromBody(
                                pageBody: $defaultOrFirstContentSection?->content ?? $page->pageBody,
                                maxWordCount: $previewMaxWordCount
                            );

                            $thisPageRenderString = str_replace(
                                search: "{{ ARTICLE_PREVIEW }}",
                                replace: $previewText . "[&hellip;]",
                                subject: $thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search: "{{ ARTICLE_URI }}",
                                replace: $page->pageRoute,
                                subject: $thisPageRenderString,
                            );

                            $thisPageRenderString = str_replace(
                                search: "{{ PAGE_URI }}",
                                replace: $page->pageRoute,
                                subject: $thisPageRenderString,
                            );

                            $publishedDate = 0;

                            if($page->publicationTimestamp !== 0) {
                                $publishedDate = $page->publicationTimestamp;
                            } else {
                                $publishedDate = $page->creationTime;
                            }

                            $thisPageRenderString = str_replace(
                                search: "{{ PUBLISHED_DATE }}",
                                replace: date("F j, Y", $publishedDate),
                                subject: $thisPageRenderString,
                            );

                            $finalStringToRender .= $thisPageRenderString;
                        }

                        $finalStringToRender = str_replace(
                            search: "{{ NUM_ITEMS }}",
                            replace: count($pagesToRender),
                            subject: $finalStringToRender,
                        );

                        return $finalStringToRender;
                    },
                    subject:$viewFileContents
                );

                return $viewFileContents;
            }
		}
	}
