<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";
	require_once __DIR__ . "/ShortcodeViewProcessor.php";

	use Exception;
	use Nox\ORM\ColumnQuery;
	use Nox\ORM\Pager;
	use Nox\ORM\ResultOrder;
	use Page\Page;
	use Page\PageData;
	use Page\PageDatas;
	use Page\PageType;
	use Page\PublicationStatus;
	use Reviews\Review;
	use Settings\Setting;
	use Settings\Settings;
	use ShortcodeParser\Processors\Exceptions\InvalidAttributeValue;
	use ShortcodeParser\Processors\Exceptions\MissingRequiredAttribute;
	use ShortcodeParser\Processors\Exceptions\ShortcodeViewFileNotFound;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;
	use System\ContentHelper;
	use System\Themes;

	/**
	 * Processor class for the get-reviews shortcode
	 */
	class GetReviews extends ShortcodeViewProcessor {

		protected static string $defaultShortcodeViewBaseFileName = "reviews.php";

		protected static array $supportedTemplateTags = [
			"{{ COLUMNS }}",
			"{{ begin ReviewItem }}",
			"{{ COLUMN_CLASS }}",
			"{{ BODY }}",
			"{{ FIRST_NAME }}",
			"{{ LAST_NAME }}",
			"{{ CITY }}",
			"{{ STATE }}",
			"{{ RATING }}",
			"{{ ID }}",
		];

		protected static array $supportedAttributes = [
			"columns",
			"truncate-body",
			"limit",
			// "offset",
			"city",
			"state",
			// "schema", // Deprecated, probably
			"show-empty-message",
			"empty-message",
			"show-header",
			"header",
			"show-empty-header",
			"shortcode-file",
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public static function getDefaultShortcodeViewBaseFileName(): string{
			return self::$defaultShortcodeViewBaseFileName;
		}

		public static function getSupportedTemplateTags(): array{
			return self::$supportedTemplateTags;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws MissingRequiredAttribute
		 * @throws InvalidAttributeValue
		 * @throws Exception
		 */
		public function runProcessor(): string{

			$reviewsLimitAttribute = $this->shortcode->getAttribute("limit");
			if ($reviewsLimitAttribute === null){
				$reviewsLimit = 10;
			}else{
				$reviewsLimit = (int) $reviewsLimitAttribute->value;
			}

			$truncateBodyAttribute = $this->shortcode->getAttribute("truncate-body");
			$truncateBody = false;
			if ($truncateBodyAttribute !== null){
				if ($truncateBodyAttribute->value === "yes" || $truncateBodyAttribute->value === "true") {
					$truncateBody = true;
				}
			}

			$cityAttribute = $this->shortcode->getAttribute("city");
			$city = null;
			if ($cityAttribute !== null && $cityAttribute->value !== ""){
				$city = $cityAttribute->value;
			}

			$stateAttribute = $this->shortcode->getAttribute("state");
			$state = null;
			if ($stateAttribute !== null && $stateAttribute->value !== ""){
				$state = $stateAttribute->value;
			}

			$showEmptyMessageAttribute = $this->shortcode->getAttribute("show-empty-message");
			$showEmptyMessage = false;
			if ($showEmptyMessageAttribute !== null){
				if ($showEmptyMessageAttribute->value === "yes" || $showEmptyMessageAttribute->value === "true"){
					$showEmptyMessage = true;
				}
			}

			$showHeaderAttribute = $this->shortcode->getAttribute("show-header");
			$showHeader = false;
			if ($showHeaderAttribute !== null){
				if ($showHeaderAttribute->value === "yes" || $showHeaderAttribute->value === "true"){
					$showHeader = true;
				}
			}

			// This header attribute is here to control if the header should still be shown
			// even when there are no reviews.
			$showEmptyHeaderAttribute = $this->shortcode->getAttribute("show-empty-header");
			$showEmptyHeader = false;
			if ($showEmptyHeaderAttribute !== null){
				if ($showEmptyHeaderAttribute->value === "yes" || $showEmptyHeaderAttribute->value === "true"){
					$showEmptyHeader = true;
				}
			}

			$emptyMessageAttribute = $this->shortcode->getAttribute("empty-message");
			$emptyMessage = null;
			if ($emptyMessageAttribute !== null && $emptyMessageAttribute->value !== ""){
				$emptyMessage = $emptyMessageAttribute->value;
			}

			$headerAttribute = $this->shortcode->getAttribute("header");
			$header = null;
			if ($headerAttribute !== null && $headerAttribute->value !== ""){
				$header = $headerAttribute->value;
			}

			// Handle columns + validation
			$columnsAttribute = $this->shortcode->getAttribute("columns");
			if ($columnsAttribute === null){
				$columns = 1;
			}else{
				$columns = (int) $columnsAttribute->value;
				if ($columns === 0 || $columns < 0){
					throw new InvalidAttributeValue("The value of 'columns' attribute must be a valid, positive integer.");
				}
			}

			// Shortcode folder
			$shortcodeViewsFolder = Themes::getCurrentThemeShortcodeViewsDirectory();

			// Is this using a custom shortcode view file?
			$shortcodeViewFileAttribute = $this->shortcode->getAttribute("shortcode-file");
			if ($shortcodeViewFileAttribute !== null){
				// Fetch the file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, $shortcodeViewFileAttribute->value);
			}else{
				// Use the default file
				$viewFile = sprintf("%s/%s", $shortcodeViewsFolder, self::$defaultShortcodeViewBaseFileName);
			}

			// Get the view file and then its contents
			$viewFileNormalizedPath = realpath($viewFile);
			if ($viewFileNormalizedPath === false){
				throw new ShortcodeViewFileNotFound(
					sprintf("No shortcode view file found at the following path: %s", $viewFile)
				);
			}

			$viewFileContents = file_get_contents($viewFileNormalizedPath);

			// Replace the template tags
			$viewFileContents = str_replace(
				search: "{{ COLUMNS }}",
				replace: $columns,
				subject: $viewFileContents,
			);

			// TODO
			// Replace this with something more scalable and customizable
			// Hard-coded Bootstrap column classes might limit us in the future
			$columnClassStub = "col-lg-"; // %s because this will be used in a string format function (sprintf)
			$classColNum = 12 / $columns;

			// Replace the template tag HTML wrapper with the necessary articles to show
			$viewFileContents = preg_replace_callback(
				pattern:"/{{ begin ReviewItem }}(.+?){{ end ReviewItem }}/ism",
				callback: function($matches) use (
					$showEmptyHeader,
					$columnClassStub,
					$classColNum,
					$city,
					$state,
					$reviewsLimit,
					$showEmptyMessage,
					$emptyMessage,
					$showHeader,
					$header,
					$truncateBody
				){
					$template = $matches[1];
					$finalStringToRender = "";

					/** @var Review[] $reviewsToRender */
					$reviewsToRender = [];

					if ($city !== null && $state !== null){
						$reviewsToRender = Review::query(
							columnQuery: (new ColumnQuery())
								->where("city","=",$city)
								->and()
								->where("state","=",$state),
							resultOrder: (new ResultOrder())
								->by("posted_timestamp","DESC"),
							pager: (new Pager(pageNumber: 1, limit: $reviewsLimit))
						);
					}elseif ($city !== null && $state === null){
						$reviewsToRender = Review::query(
							columnQuery: (new ColumnQuery())
								->where("city","=",$city),
							resultOrder: (new ResultOrder())
								->by("posted_timestamp","DESC"),
							pager: (new Pager(pageNumber: 1, limit: $reviewsLimit))
						);
					}elseif ($city === null && $state !== null){
						$reviewsToRender = Review::query(
							columnQuery: (new ColumnQuery())
								->where("state","=",$state),
							resultOrder: (new ResultOrder())
								->by("posted_timestamp","DESC"),
							pager: (new Pager(pageNumber: 1, limit: $reviewsLimit))
						);
					}else{
						// Both are null
						$reviewsToRender = Review::query(
							resultOrder: (new ResultOrder())
								->by("posted_timestamp","DESC"),
							pager: (new Pager(pageNumber: 1, limit: $reviewsLimit))
						);
					}


					if (count($reviewsToRender) === 0) {
						if ($showEmptyHeader === true) {
							if ($header !== null) {
								$finalStringToRender .= sprintf('<div class="col-12"><h2>%s</h2></div>', $header);
							} else {
								$finalStringToRender .= '<div class="col-12"><h2>Reviews and Testimonials</h2></div>';
							}
						}

						if ($showEmptyMessage === true) {
							// Check if the "empty-message" attribute is null
							if ($emptyMessage !== null) {
								$finalStringToRender .= sprintf('<div class="col-12"><p>%s</p></div>', $emptyMessage);
							} else {
								// Check if the Setting for the custom empty message is null
								$emptyMessageSetting = Setting::getSettingValue(Settings::NO_REVIEWS_MESSAGE->value);
								if (!empty($emptyMessageSetting)) {
									$finalStringToRender .= $emptyMessageSetting;
								} else {
									$finalStringToRender .= '<div class="col-12"><p>There are currently no reviews or testimonials; check back soon!</p></div>';
								}
							}
						}
					}else {
						if ($showHeader === true) {
							if ($header !== null) {
								$finalStringToRender .= sprintf('<div class="col-12"><h2>%s</h2></div>', $header);
							}else{
								$finalStringToRender .= '<div class="col-12"><h2>Reviews and Testimonials</h2></div>';
							}
						}

						/**
						 * @var int $index
						 * @var Review $review
						 */
						foreach ($reviewsToRender as $index => $review) {
							// Make a copy of the string
							$thisPageRenderString = $template;

							// Make all template tag replacements

							$thisPageRenderString = str_replace(
								search: "{{ COLUMN_CLASS }}",
								replace: sprintf("%s%s", $columnClassStub, $classColNum),
								subject: $thisPageRenderString,
							);

							if ($truncateBody === true) {

								$thisPageRenderString = str_replace(
									search: "{{ BODY }}",
									replace: substr($review->body, 0, 120) . " [...]",
									subject: $thisPageRenderString,
								);
							} else {
								$thisPageRenderString = str_replace(
									search: "{{ BODY }}",
									replace: $review->body,
									subject: $thisPageRenderString,
								);
							}

							$thisPageRenderString = str_replace(
								search: "{{ FIRST_NAME }}",
								replace: $review->firstName,
								subject: $thisPageRenderString,
							);

							$thisPageRenderString = str_replace(
								search: "{{ LAST_NAME }}",
								replace: $review->lastName,
								subject: $thisPageRenderString,
							);

							$thisPageRenderString = str_replace(
								search: "{{ CITY }}",
								replace: $review->city,
								subject: $thisPageRenderString,
							);

							$thisPageRenderString = str_replace(
								search: "{{ STATE }}",
								replace: $review->state,
								subject: $thisPageRenderString,
							);

							$thisPageRenderString = str_replace(
								search: "{{ RATING }}",
								replace: $review->rating,
								subject: $thisPageRenderString,
							);

							$thisPageRenderString = str_replace(
								search: "{{ ID }}",
								replace: $review->id,
								subject: $thisPageRenderString,
							);

							$finalStringToRender .= $thisPageRenderString;
						}
					}

					return $finalStringToRender;
				},
				subject:$viewFileContents
			);

			return $viewFileContents;
		}
	}