<?php
	namespace ShortcodeParser\Processors;

	require_once __DIR__ . "/Processor.php";

	use Page\Page;
	use Page\PageDatas;
	use ShortcodeParser\Processors\Exceptions\NoPageCurrentlyRendered;
	use ShortcodeParser\Shortcode;
	use ShortcodeParser\ShortcodeTypes;

	/**
	 * Processor class for the state-name shortcode.
	 * Historically, this shortcode allowed a shorthand attribute to be specified to tell the
	 * processor we want the shorthand state name. This is for compatibility from CMS v3. In modern times,
	 * we now have the state-name-shorthand shortcode.
	 */
	class GetStateName extends Processor {

		protected static array $supportedAttributes = [
			"map-safe",
			"short-hand",
		];

		public static function getSupportedAttributeNames(): array{
			return self::$supportedAttributes;
		}

		public function __construct(
			public Shortcode $shortcode
		){}

		/**
		 * @throws NoPageCurrentlyRendered
		 */
		public function runProcessor(): string{
			$shouldFormatToMapSafe = $this->shortcode->getAttribute("map-safe") !== null;

			// Fetch the current page, if any
			$currentPage = Page::getCurrentPage();
			if ($currentPage !== null){
				$needsShorthand = $this->shortcode->getAttribute("short-hand") !== null;

				// Get the proper state name format requested
				if ($needsShorthand){
					$stateNameDatas = $currentPage->getPageDatas(PageDatas::STATE_NAME_SHORTHAND);
				}else {
					$stateNameDatas = $currentPage->getPageDatas(PageDatas::STATE_NAME);
				}

				if (empty($stateNameDatas)){
					// No state name for this page
					// Return blank
					return "";
				}else{
					// Get the first result
					$stateNameData = $stateNameDatas[0];
					$stateName = $stateNameData->value;

					// Does it need to be map-safe?
					if ($shouldFormatToMapSafe){
						$stateName = preg_replace(
							pattern:"@\s@",
							replacement: "+",
							subject: $stateName,
						);
					}

					return $stateName;
				}
			}else{
				throw new NoPageCurrentlyRendered(
					sprintf(
						"Failed to run %s shortcode. There is no current Uplift CMS page being rendered.",
						ShortcodeTypes::GET_STATE_NAME->value,
					)
				);
			}
		}
	}
