<?php
	namespace ShortcodeParser;

	use Exception;
	use ImageGallery\NoImageGalleryFound;
	use MonologWrapper\MonologWrapper;
	use ShortcodeParser\Processors\BaseURL;
	use ShortcodeParser\Processors\GenerateSitemap;
	use ShortcodeParser\Processors\GetArticles;
	use ShortcodeParser\Processors\GetBreadcrumbs;
	use ShortcodeParser\Processors\GetCitiesList;
	use ShortcodeParser\Processors\GetPageName;
	use ShortcodeParser\Processors\GetServicesList;
	use ShortcodeParser\Processors\GetCityName;
	use ShortcodeParser\Processors\GetCompanyCity;
	use ShortcodeParser\Processors\GetCompanyFaxNumber;
	use ShortcodeParser\Processors\GetCompanyName;
	use ShortcodeParser\Processors\GetCompanyPhoneNumber;
	use ShortcodeParser\Processors\GetCompanyPostal;
	use ShortcodeParser\Processors\GetCompanyState;
	use ShortcodeParser\Processors\GetCompanyStreet;
	use ShortcodeParser\Processors\GetGallery;
	use ShortcodeParser\Processors\GetGoogleMapsAPIKey;
	use ShortcodeParser\Processors\GetPageAttributePageValue;
	use ShortcodeParser\Processors\GetPages;
	use ShortcodeParser\Processors\GetProjects;
	use ShortcodeParser\Processors\GetRandomPages;
	use ShortcodeParser\Processors\GetRecentArticles;
	use ShortcodeParser\Processors\GetRecentProjects;
	use ShortcodeParser\Processors\GetReviews;
	use ShortcodeParser\Processors\GetStateName;
	use ShortcodeParser\Processors\GetStateNameShorthand;
	use ShortcodeParser\Processors\IncludeFile;
	use ShortcodeParser\Processors\URL;
	use UnhandledMatchError;

	class ShortcodeParser{


		/**
		 * Parses the shortcodes in an entire content string. Replaces them with their string equivalents or will
		 * replace them with the exception error message, if there is one.
		 * @param string $content
		 * @return string
		 */
		public function parse(string $content): string
		{
			return preg_replace_callback("/{{.+?}}/ism", function (array $match) {
				$codeString = trim($match[0]);

				try {
					$processedString = $this->parseCodeString($codeString);
				} catch (InvalidShortcode $e) {
					return "Error with shortcode. Error message is: {$e->getMessage()}";
				} catch (ShortcodeSyntaxError $e) {
					return "Shortcode syntax error: {$e->getMessage()}";
				} catch (Processors\Exceptions\InvalidAttributeValue $e) {
					return "Invalid attribute value: {$e->getMessage()}.";
				} catch (Processors\Exceptions\MissingRequiredAttribute $e) {
					return "Missing required attribute: {$e->getMessage()}.";
				} catch (NoImageGalleryFound $e){
					return "Image gallery error: {$e->getMessage()}.";
				} catch (Exception $e){
					return "Error: {$e->getMessage()}. \n\n<br><br>Trace: <br>\n{$e->getTraceAsString()}";
				}

				return $processedString;
			}, $content);
		}

		/**
		 * @throws InvalidShortcode
		 * @throws NoImageGalleryFound
		 * @throws Processors\Exceptions\MissingRequiredAttribute
		 * @throws ShortcodeSyntaxError
		 * @throws Processors\Exceptions\InvalidAttributeValue
		 * @throws Exception
		 */
		public function parseCodeString(string $codeString): string{
			$parser = new ShortcodeLexicalParser();
			$shortcode = $parser->getShortcodeFromString($codeString);

			return $this->processShortcode($shortcode);
		}

		/**
		 * Processes the shortcode object and gets the resulting string after it has been processed.
		 * @param Shortcode $shortcode
		 * @return string
		 * @throws NoImageGalleryFound
		 * @throws Processors\Exceptions\InvalidAttributeValue
		 * @throws Processors\Exceptions\MissingRequiredAttribute
		 * @throws Exception
		 */
		public function processShortcode(Shortcode $shortcode): string{
			try {
				return match ($shortcode->name) {
					ShortcodeTypes::BASE_URL->value, ShortcodeTypes::BASE_URL_ALT->value => (new BaseURL($shortcode))->runProcessor(),
					ShortcodeTypes::URL->value => (new URL($shortcode))->runProcessor(),
					ShortcodeTypes::GET_ARTICLES->value => (new GetArticles($shortcode))->runProcessor(),
					ShortcodeTypes::GET_PROJECTS->value, ShortcodeTypes::GET_PROJECTS_ALT->value => (new GetProjects($shortcode))->runProcessor(),
					ShortcodeTypes::GET_RECENT_ARTICLES->value => (new GetRecentArticles($shortcode))->runProcessor(),
					ShortcodeTypes::GET_RECENT_PROJECTS->value => (new GetRecentProjects($shortcode))->runProcessor(),
					ShortcodeTypes::INCLUDE_FILE->value => (new IncludeFile($shortcode))->runProcessor(),
					ShortcodeTypes::GET_CITY_NAME->value, ShortcodeTypes::GET_CITY_NAME_ALT->value => (new GetCityName($shortcode))->runProcessor(),
					ShortcodeTypes::GET_STATE_NAME->value, ShortcodeTypes::GET_STATE_NAME_ALT->value => (new GetStateName($shortcode))->runProcessor(),
					ShortcodeTypes::GET_STATE_NAME_SHORTHAND->value => (new GetStateNameShorthand($shortcode))->runProcessor(),
					ShortcodeTypes::COMPANY_NAME->value, ShortcodeTypes::COMPANY_NAME_ALT->value => (new GetCompanyName($shortcode))->runProcessor(),
					ShortcodeTypes::COMPANY_STATE->value, ShortcodeTypes::COMPANY_STATE_ALT->value => (new GetCompanyState($shortcode))->runProcessor(),
					ShortcodeTypes::COMPANY_CITY->value, ShortcodeTypes::COMPANY_CITY_ALT->value => (new GetCompanyCity($shortcode))->runProcessor(),
					ShortcodeTypes::COMPANY_POSTAL->value, ShortcodeTypes::COMPANY_POSTAL_ALT->value => (new GetCompanyPostal($shortcode))->runProcessor(),
					ShortcodeTypes::COMPANY_STREET->value, ShortcodeTypes::COMPANY_STREET_ALT->value => (new GetCompanyStreet($shortcode))->runProcessor(),
					ShortcodeTypes::GET_REVIEWS->value, ShortcodeTypes::GET_REVIEWS_ALT->value => (new GetReviews($shortcode))->runProcessor(),
					ShortcodeTypes::BREADCRUMBS->value => (new GetBreadcrumbs($shortcode))->runProcessor(),
					ShortcodeTypes::GET_LIST_OF_CITIES->value => (new GetCitiesList($shortcode))->runProcessor(),
					ShortcodeTypes::GET_LIST_OF_SERVICES->value => (new GetServicesList($shortcode))->runProcessor(),
					ShortcodeTypes::PHONE_NUMBER->value, ShortcodeTypes::PHONE_NUMBER_ALT->value => (new GetCompanyPhoneNumber($shortcode))->runProcessor(),
					ShortcodeTypes::FAX_NUMBER->value, ShortcodeTypes::FAX_NUMBER_ALT->value => (new GetCompanyFaxNumber($shortcode))->runProcessor(),
					ShortcodeTypes::GENERATE_SITEMAP->value, ShortcodeTypes::GENERATE_SITEMAP_ALT->value => (new GenerateSitemap($shortcode))->runProcessor(),
					ShortcodeTypes::GET_RANDOM_PAGES->value => (new GetRandomPages($shortcode))->runProcessor(),
					ShortcodeTypes::GOOGLE_MAPS_API_KEY->value, ShortCodeTypes::GOOGLE_MAPS_API_KEY_ALT->value => (new GetGoogleMapsAPIKey($shortcode))->runProcessor(),
					ShortcodeTypes::TEST->value => "",
					ShortcodeTypes::GET_GALLERY->value => (new GetGallery($shortcode))->runProcessor(),
					ShortcodeTypes::GET_PAGE_ATTRIBUTE_VALUE->value => (new GetPageAttributePageValue($shortcode))->runProcessor(),
					ShortcodeTypes::GET_PAGES->value => (new GetPages($shortcode))->runProcessor(),
					ShortcodeTypes::GET_PAGE_NAME->value => (new GetPageName($shortcode))->runProcessor(),
				};
			}catch(UnhandledMatchError $e){
				// Just log it
				$logger = MonologWrapper::getLogger();
				$logger->error($e->getMessage());
				return "";
			}
		}
	}
