<?php
	namespace System;

	use ShortcodeParser\ShortcodeParser;

	class ContentHelper{

		/**
		 * Default set of characters to use when generating gibberish
		 */
		const DEFAULT_GIBBERISH_CHARACTERS = "abcdefghijklmnopqrstuvwxyz0123456789";

		/**
		 * Makes a valid preview from a page body. Removes unnecessary templating tags and shortcodes before then
		 * parsing the shortcodes left and returning a preview of the content.
		 * @param string $pageBody
		 * @return string
		 */
		public static function makePreviewFromBody(
			string $pageBody,
			int $maxWordCount = 45,
			int $maxCharacterCounter = 900,
			bool $allowLineBreaks = false,
		): string{

			// Remove breadcrumbs shortcode if there is one
			$pageBody = preg_replace("@{{\s*breadcrumbs\s*}}@", "", $pageBody);

			// Remove section template tags if there are any
			$pageBody = preg_replace("@{% (begin|end) [a-zA-Z_0-9]+ %}@", "", $pageBody);

			// Remove get-recent-articles
			$pageBody = preg_replace("@{{\s*get-recent-articles.+}}@", "", $pageBody);

			// Remove get-recent-projects
			$pageBody = preg_replace("@{{\s*get-recent-projects.+}}@", "", $pageBody);

			// Remove get-articles
			$pageBody = preg_replace("@{{\s*get-articles.+}}@", "", $pageBody);

			$shortcodeParser = new ShortcodeParser();
			$body = $shortcodeParser->parse($pageBody);

			// Remove h1
			$body = preg_replace("/<h1.*?>.+?<\/h1>/ism", "", $body);

			// Remove h2
			$body = preg_replace("/<h2.*?>.+?<\/h2>/ism", "", $body);

			// Strip all tags
			$allowedTags = "";

			if ($allowLineBreaks === true){
				$allowedTags = "<br>";
			}

			$body = trim(strip_tags($body, $allowedTags));

			if (strlen($body) > $maxCharacterCounter){
				$body = substr($body, 0, $maxCharacterCounter - 1);
			}

			// Truncate words
			$words = explode(" ", $body);

			$bodyPreview = "";

			foreach($words as $index=>$word){
				if ($index > $maxWordCount){
					break;
				}
				$bodyPreview .= $word  . " ";
			}

			return $bodyPreview;
		}

		public static function generateGibberish(
			int $length,
			?string $charactersToUse = null,
		): string{

			if ($charactersToUse === null){
				$charactersToUse = self::DEFAULT_GIBBERISH_CHARACTERS;
			}

			$counter = 0;
			$buffer = "";
			$allowedCharactersLength = strlen($charactersToUse);

			while ($counter < $length){
				try {
					$shouldUpper = random_int(0, 1);
				}catch(\Exception){
					$shouldUpper = false;
				}
				if ($shouldUpper) {
					$buffer .= $charactersToUse[rand(0, $allowedCharactersLength-1)];
				}else{
					$buffer .= strtoupper($charactersToUse[rand(0, $allowedCharactersLength-1)]);
				}

				++$counter;
			}

			return $buffer;
		}
	}