<?php
    namespace System;

	use Exception;

	class HttpHelper{
		public static function getIPFromRequest(): string{
			if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
				// IP is from shared internet
				return $_SERVER['HTTP_CLIENT_IP'];
			} elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
				// IP is from a proxy
				return $_SERVER['HTTP_X_FORWARDED_FOR'];
			} elseif (!empty($_SERVER['REMOTE_ADDR'])) {
				// IP is from a remote address
				return $_SERVER['REMOTE_ADDR'];
			}

			return "";
		}

		/**
		 * Fetches the current website's base, full-qualified URL. Does not
		 * include a forward slash at the end.
		 * @return string
		 * @throws Exception
		 */
		public static function getWebsiteBaseURL(): string{
			if (php_sapi_name() !== "cli") {
				$isSSLOn = !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off';

				// GoDaddy, Amazon servers, etc
				// Sometimes they use the following method to determine if SSL is on for some reason...
				if ($isSSLOn === false) {
					if (!empty($_SERVER['HTTP_X_FORWARDED_PROTO'])) {

						// These external shared servers forward requests through multiple other servers
						// So we have to check if the original forwarded request was secure.
						$protocolForwarded = $_SERVER['HTTP_X_FORWARDED_PROTO'];
						if (trim($protocolForwarded) === "https") {
							$isSSLOn = true;
						}
					}
				}

				return ($isSSLOn ? "https://" : "http://") . $_SERVER["HTTP_HOST"];
			}else{
				// Try and fetch it from the command line arguments
				global $argv;
				$baseURL = "";

				foreach($argv as $arg){
					$keyValue = explode("=", $arg);
					if ($keyValue[0] === "BASE_URL"){
						$baseURL = $keyValue[1];
					}
				}

				if (empty($baseURL)) {
					throw new Exception("Cannot fetch the base URL of website because method was called from the command line. Add BASE_URL=https://example.com to add the BASE_URL via the command line.");
				}else{
					return $baseURL;
				}
			}
		}

		public static function getURI(): string{
			return $_SERVER['REQUEST_URI'];
		}

		public static function getFirstHeaderValue(string $headerName): ?string{
			$httpHeaders = getallheaders();

			foreach($httpHeaders as $arrayHeaderName=>$headerValue) {
				if (strtolower($arrayHeaderName) === strtolower($headerName)) {
					return $headerValue;
				}
			}

			return null;
		}

		/**
		 * Fetches a safe query string from the current request.
		 * @return string
		 */
		public static function getQueryString(): string{
			$queryString = $_SERVER['QUERY_STRING'];
			if (strlen($queryString) > 0) {
				$queryStringParts = explode("&", $queryString);
				$sanitizedParts = array_map(fn($str) => htmlspecialchars($str), $queryStringParts);
				return implode("&", $sanitizedParts);
			}

			return "";
		}
	}