<?php
	namespace System;

	use Composer\InstalledVersions;
	use Composer\Semver\Semver;
	use Composer\Semver\VersionParser;
	use System\Exceptions\ExtensionVersionMismatch;
	use System\Exceptions\MissingRequiredExtension;
	use System\Exceptions\MissingRequiredPackage;
	use System\Exceptions\PackageVersionMismatch;
	use System\Exceptions\PHPVersionsIncompatible;

	/**
	 * Utility class to check requirements against packages, extensions, and PHP version listed in the 'require' block of a composer.json file.
	 */
	class PlatformRequirements{
		/**
		 * Uses a JSON-decoded require array from a composer file to check if the current environment running this code can support the requirements listed in the composer.json require block.
		 * @param array $composerRequireArray
		 * @return void
		 * @throws MissingRequiredExtension
		 * @throws PHPVersionsIncompatible
		 */
		public static function assertPlatformRequirements(array $composerRequireArray): void{
			foreach($composerRequireArray as $packageName=>$versionRequirement){
				$packageName = strtolower($packageName);
				if ($packageName === "php"){
					self::assertCurrentPHPVersionIsCompatible($versionRequirement);
				}elseif (str_starts_with($packageName, "ext-")){
					// PHP extension
					$extensionName = substr($packageName, 4);
					self::assertExtensionExists($extensionName, $versionRequirement);
				}else{
					// Normal composer-installed vendor package

					// Commented out on 12/15/2022
					// New Vendor packages are packaged with the new update/payload - so of course they might be mismatched
					// with what the current system has. Ignore a vendor package check.

					// self::assertVendorPackageIsInstalledAndHasCorrectVersion($packageName, $versionRequirement);
				}
			}
		}

		/**
		 * @throws PHPVersionsIncompatible
		 */
		public static function assertCurrentPHPVersionIsCompatible(string $versionConstraint): void{
			$currentPHPVersion = PHP_VERSION;
			if (!Semver::satisfies(PHP_VERSION, $versionConstraint)){
				throw new PHPVersionsIncompatible("The PHP version this system is running is " . $currentPHPVersion . " but an update requires " . $versionConstraint);
			}
		}

		/**
		 * PHP has no accurate way of getting the version number of an extension. phpversion(string $extensionName) will sometimes
		 * pull the current PHP version for an extension - because the extension isn't reporting its version to PHP. We can
		 * only assert if the extension is there or not.
		 * @throws MissingRequiredExtension
		 */
		public static function assertExtensionExists(string $extensionName): void{
			$extensionVersion = phpversion($extensionName);
			if ($extensionVersion === false){
				throw new MissingRequiredExtension($extensionName . " is required for an update but doesn't exist on this system.");
			}
		}

		/**
		 * @throws MissingRequiredPackage
		 * @throws PackageVersionMismatch
		 */
		public static function assertVendorPackageIsInstalledAndHasCorrectVersion(string $packageName, string $versionConstraint): void{
			if (!InstalledVersions::isInstalled($packageName)){
				throw new MissingRequiredPackage($packageName . " is missing and the update cannot process without it.");
			}else{
				if (!InstalledVersions::satisfies(new VersionParser(), $packageName, $versionConstraint)){
					throw new PackageVersionMismatch("The vendor package $packageName has a version that doesn't satisfy the constraint $versionConstraint");
				}
			}
		}
	}