<?php

	namespace TemplateManager\PageLayouts;

	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Request;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;

	#[Controller]
	#[RouteBase("/uplift/theme-manager/page-layouts")]
	class PageLayoutsController extends BaseController{

		/**
		 * Fetches the PageLayoutSectionDefinition for the provided layout file path.
		 * The PageLayoutSectionDefinition will eager-load the individual PageLayoutSection objects
		 * @return JSONResult
		 */
		#[Route("GET", "/sections-definition")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		#[UseJSON]
		public function getPageLayoutSectionDefinition(): JSONResult
		{
			$filePath = $_GET['filePath'] ?? "";

			if (empty(trim($filePath))){
				return new JSONError("File path cannot be empty.");
			}

			try {
				$pageLayoutSectionProvider = new PageLayoutSectionsProvider($filePath);
			}catch(\Exception $ex){
				return new JSONError($ex->getMessage());
			}

			try{
				$sectionDefinition = $pageLayoutSectionProvider->getSectionDefinition();
			}catch(\Exception $ex){
				return new JSONError($ex->getMessage());
			}

			return new JSONSuccess([
				"pageLayoutSectionDefinition"=>$sectionDefinition
			]);
		}

		/**
		 * This route will create a page layout section for the provided section definition Id. However,
		 * if no section definition Id is provided then this route will check if layoutFilePath was provided and
		 * try and create a page layout section definition first, then the page layout section.
		 * @param Request $request
		 * @return JSONResult
		 */
		#[Route("POST", "/sections-definition/sections")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		#[UseJSON]
		#[ProcessRequestBody]
		public function createPageLayoutSection(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$pageLayoutSectionDefinitionId = (int)$payload->getTextPayloadNullable("pageLayoutSectionDefinitionId")?->contents;
			try{
				$sectionName = $payload->getTextPayload("sectionName");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			if ($pageLayoutSectionDefinitionId > 0){
				// Add the section to the section definition

				try{
					$newSection = PageLayoutsService::createPageLayoutSectionForDefinition(
						$pageLayoutSectionDefinitionId,
						$sectionName->contents
					);
				}catch(\Exception $ex){
					return new JSONError($ex->getMessage());
				}

				return new JSONSuccess([
					"section"=>$newSection
				]);
			}else{
				// Was the full file path set?
				$pageLayoutFullFilePath = $payload->getTextPayloadNullable("pageLayoutFullFilePath");
				if ($pageLayoutFullFilePath === null){
					return new JSONError("Cannot create a new page section with either a parameter for: pageLayoutSectionDefinitionId or pageLayoutFullFilePath. Use pageLayoutFullFilePath if this is creating a new section for a section definition which does not exist.");
				}

				// Create the definition
				try{
					$definition = PageLayoutsService::createPageLayoutDefinitionIfNotExists($pageLayoutFullFilePath->contents);
				}catch(\Exception $ex){
					return new JSONError($ex->getMessage());
				}

				// Create the section
				try{
					$newSection = PageLayoutsService::createPageLayoutSectionForDefinition(
						$definition->id,
						$sectionName->contents
					);
				}catch(\Exception $ex){
					return new JSONError($ex->getMessage());
				}

				return new JSONSuccess([
					"definition"=>$definition,
					"section"=>$newSection,
				]);
			}
		}

		/**
		 * Updates an individual page layout section
		 * @param Request $request
		 * @return JSONResult
		 */
		#[Route("PATCH", "@\/sections-definition\/sections\/(?<pageLayoutSectionId>\d+)@", true)]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		#[UseJSON]
		#[ProcessRequestBody]
		public function editPageLayoutSection(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$pageLayoutSectionId = (int)$request->getParameter("pageLayoutSectionId");
			$newPageLayoutSectionName = $payload->getTextPayloadNullable("newSectionName")?->contents ?? null;
			$isDefaultContentSection = (int)($payload->getTextPayloadNullable("isDefaultContentSection")?->contents ?? "0");

			try{
				$section = PageLayoutsService::editPageLayoutSection(
					$pageLayoutSectionId,
					$newPageLayoutSectionName,
					$isDefaultContentSection
				);

				return new JSONSuccess([
					"section"=>$section
				]);
			}catch(\Exception $ex){
				return new JSONError($ex->getMessage());
			}
		}

		/**
		 * Deletes an individual page layout section. If this was the last page layout section,
		 * then the page layout definition is also removed.
		 * @param Request $request
		 * @return JSONResult
		 */
		#[Route("DELETE", "@\/sections-definition\/sections\/(?<pageLayoutSectionId>\d+)@", true)]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		#[UseJSON]
		public function deletePageLayoutSection(Request $request): JSONResult
		{
			$pageLayoutSectionId = (int)$request->getParameter("pageLayoutSectionId");

			try{
				PageLayoutsService::deletePageLayoutSection(
					$pageLayoutSectionId,
				);
				return new JSONSuccess();
			}catch(\Exception $ex){
				return new JSONError($ex->getMessage());
			}
		}
	}