<?php
	namespace TemplateManager\PageLayouts;

	use Exception;
	use InvalidArgumentException;
	use Nox\ORM\ColumnQuery;
	use Page\Page;
	use Page\PageContentSection;
	use System\Themes;

	class PageLayoutsService{
		/**
		 * Creates a page layout definition for the provided page layout file path, if one does not exist.
		 * @throws Exception
		 * @throws InvalidArgumentException
		 */
		public static function createPageLayoutDefinitionIfNotExists(string $fullLayoutFilePath): PageLayoutSectionsDefinition{
			$layoutPathStartingWithUpliftData = PageLayoutSectionsProvider::getLayoutFilePathRelativeToProjectRoot($fullLayoutFilePath);

			// Check if a definition already exists for that layout file
			/** @var PageLayoutSectionsDefinition $layoutDefinition */
			$layoutDefinition = PageLayoutSectionsDefinition::queryOne(
				columnQuery: (new ColumnQuery())
					->where("layout_file_name", "=", $layoutPathStartingWithUpliftData)
			);

			if ($layoutDefinition !== null){
				throw new InvalidArgumentException("A PageLayoutSectionDefinition already exists for the file path $fullLayoutFilePath");
			}else{
				// Create and return it
				$newSectionDefinition = new PageLayoutSectionsDefinition();
				$newSectionDefinition->layoutFileName = $layoutPathStartingWithUpliftData;
				$newSectionDefinition->save();
				return $newSectionDefinition;
			}
		}

		/**
		 * Creates a page layout section for the provided definition Id, if one with the same name on that
		 * definition doesn't already exist.
		 * @param int $pageLayoutSectionDefinitionId
		 * @param string $sectionName
		 * @return void
		 * @throws InvalidArgumentException
		 */
		public static function createPageLayoutSectionForDefinition(
			int $pageLayoutSectionDefinitionId,
			string $sectionName,
		): PageLayoutSection{
			/** @var ?PageLayoutSectionsDefinition $definition */
			$definition = PageLayoutSectionsDefinition::fetch($pageLayoutSectionDefinitionId);

			if ($definition === null){
				throw new InvalidArgumentException("No definition found with Id $pageLayoutSectionDefinitionId");
			}

			// Find a section with the same name
			/** @var ?PageLayoutSection $existingSectionWithSameName */
			$existingSectionWithSameName = PageLayoutSection::queryOne(
				columnQuery: (new ColumnQuery())
					->where("section_name", "=",$sectionName)
					->and()
					->where("page_layout_section_definition_id", "=", $pageLayoutSectionDefinitionId)
			);

			if ($existingSectionWithSameName !== null){
				throw new InvalidArgumentException("A section with that name already exists for this layout.");
			}

			if (strlen(trim($sectionName)) === 0){
				throw new InvalidArgumentException("New section name cannot be blank.");
			}

			$newSection = new PageLayoutSection();
			$newSection->pageLayoutSectionDefinitionId = $pageLayoutSectionDefinitionId;
			$newSection->sectionName = $sectionName;
			$newSection->save();

			return $newSection;
		}

		/**
		 * Updates an existing page layout section. If null is provided to a value, then it will not be affected
		 */
		public static function editPageLayoutSection(
			int $pageLayoutSectionId,
			?string $newPageLayoutSectionName,
			int $isDefaultContentSection
		): PageLayoutSection{
			/** @var ?PageLayoutSection $existingSection */
			$existingSection = PageLayoutSection::fetch($pageLayoutSectionId);

			if ($existingSection === null){
				throw new InvalidArgumentException("No page layout section found with Id $pageLayoutSectionId");
			}

			// Check if the name is being set, then check if there is another section with the same section definition Id
			// with the attempted new name
			if ($newPageLayoutSectionName !== null){
				$existingSectionWithSameNameInSameDefinition = PageLayoutSection::queryOne(
					columnQuery: (new ColumnQuery())
						->where("section_name", "=", $newPageLayoutSectionName)
						->and()
						->where("page_layout_section_definition_id", "=", $existingSection->pageLayoutSectionDefinitionId)
						->and()
						->where("id", "!=", $pageLayoutSectionId)
				);

				if ($existingSectionWithSameNameInSameDefinition !== null){
					throw new InvalidArgumentException("An existing section in the same page layout exists with the new name you are trying to set. Sections in the same page layout definition cannot share names.");
				}else{
					// Store the original section name
					$originalSectionName = $existingSection->sectionName;

					// Update the name of the page layout section
					$existingSection->sectionName = $newPageLayoutSectionName;

					// Clamp the default content section flag
					if ($isDefaultContentSection < 0 || $isDefaultContentSection > 1){
						$isDefaultContentSection = 0;
					}

					// Update the default content section flag
					$existingSection->isDefaultContentSection = $isDefaultContentSection;

					// Find all page's that have the layout file name that matches this $existingSection's
					// layout file name. Strip everything from /layouts directory and behind it from the file name
					// of the existing section
					/** @var ?PageLayoutSectionsDefinition $sectionDefinition */
					$sectionDefinition = PageLayoutSectionsDefinition::queryOne(
						columnQuery: (new ColumnQuery())
							->where("id", "=", $existingSection->pageLayoutSectionDefinitionId)
					);

					if ($sectionDefinition !== null){
						$layoutsDirectory = ltrim(Themes::getCurrentThemeDirectoryURI() . "/layouts", "/");
						$layoutBaseName = ltrim(str_replace($layoutsDirectory, "", $sectionDefinition->layoutFileName), "/");
						$layoutBaseNameNoExtension = preg_replace("@\..+$@", "", $layoutBaseName);

						// Find all page's that use that layout
						/** @var Page[] $pagesUsingThatLayout */
						$pagesUsingThatLayout = Page::query(
							columnQuery: (new ColumnQuery())
								->where("pageLayout", "=", $layoutBaseNameNoExtension)
						);

						// Get a list of all of their Ids
						$pageIds = [];
						foreach ($pagesUsingThatLayout as $page){
							$pageIds[] = $page->id;
						}

						// Query all page content sections with those page Ids and sectionName $originalSectionName
						/** @var PageContentSection[] $existingPageContentSections */
						$existingPageContentSections = PageContentSection::query(
							columnQuery: (new ColumnQuery())
								->where("page_id", "IN", sprintf("(%s)", implode(",", $pageIds)))
								->and()
								->where("section_name", "=", $originalSectionName)
						);

						// Update all of their section names
						foreach ($existingPageContentSections as $pageContentSection){
							$pageContentSection->sectionName = $newPageLayoutSectionName;
						}

						PageContentSection::saveAll($existingPageContentSections);

						// If this layout section is now the default content section, unflag any other
						// layout sections that are for the same layout file
						if ($isDefaultContentSection === 1){

							/** @var PageLayoutSection[] $pageLayoutSectionsWithDefaultContentFlag */
							$pageLayoutSectionsWithDefaultContentFlag = PageLayoutSection::query(
								columnQuery: (new ColumnQuery())
									->where("page_layout_section_definition_id", "=", $existingSection->pageLayoutSectionDefinitionId)
									->and()
									->where("is_default_content_section", "=", 1)
									->and()
									->where("id", "!=", $existingSection->id)
							);

							foreach($pageLayoutSectionsWithDefaultContentFlag as $pageLayoutSection){
								$pageLayoutSection->isDefaultContentSection = 0;
								$pageLayoutSection->save();
							}
						}
					}
				}
			}

			// Save and return
			$existingSection->save();
			return $existingSection;
		}

		/**
		 * Deletes the individual page layout section. If this was the last section and the page layout section
		 * definition has no more sections, then the definition is also deleted.
		 * @param int $pageLayoutSectionId
		 * @return void
		 */
		public static function deletePageLayoutSection(int $pageLayoutSectionId): void{
			/** @var ?PageLayoutSection $existingSection */
			$existingSection = PageLayoutSection::fetch($pageLayoutSectionId);

			$sectionDefinitionId = $existingSection->pageLayoutSectionDefinitionId;

			if ($existingSection === null){
				throw new InvalidArgumentException("No page layout section found with Id $pageLayoutSectionId");
			}

			$existingSection->delete();

			/** @var ?PageLayoutSectionsDefinition $definition */
			$definition = PageLayoutSectionsDefinition::fetch($sectionDefinitionId);

			// It has to exist... so don't check for null
			$definition->loadSections();
			if (count($definition->pageLayoutSections) === 0){
				// Delete the definition
				$definition->delete();
			}
		}
	}