<?php

	namespace TemplateManager;

	use Accounts\Account;
	use Accounts\Attributes\RequireLogin;
	use Accounts\Attributes\RequirePermission;
	use ActivityLogs\ActivityLog;
	use ActivityLogs\ActivityLogCategories;
	use Exception;
	use FileSystemUtilities\exceptions\InvalidFolderName;
	use FileSystemUtilities\exceptions\NewDirectoryWithSameNameExists;
	use FileSystemUtilities\exceptions\PathDoesntExist;
	use FileSystemUtilities\FSFile;
	use Nox\Http\Attributes\ProcessRequestBody;
	use Nox\Http\Attributes\UseJSON;
	use Nox\Http\Exceptions\NoPayloadFound;
	use Nox\Http\JSON\JSONError;
	use Nox\Http\JSON\JSONResult;
	use Nox\Http\JSON\JSONSuccess;
	use Nox\Http\Redirect;
	use Nox\Http\Request;
	use Nox\ORM\ColumnQuery;
	use Nox\RenderEngine\Renderer;
	use Nox\Router\Attributes\Controller;
	use Nox\Router\Attributes\Route;
	use Nox\Router\Attributes\RouteBase;
	use Nox\Router\BaseController;
	use Roles\PermissionCategories;
	use ScssPhp\ScssPhp\Exception\SassException;
	use Settings\Setting;
	use System\System;
	use System\Themes;
	use TemplateManager\Exceptions\SassCLIException;
	use Uplift\Exceptions\DirectoryDoesntExist;
	use Uplift\Exceptions\EmptyValue;
	use Uplift\Exceptions\FileAlreadyExists;
	use Uplift\Exceptions\FileDoesntExist;
	use Uplift\Exceptions\FilePermissionError;
	use Uplift\Exceptions\IllegalDirectoryPath;
	use Uplift\Exceptions\IllegalFileName;
	use Uplift\Exceptions\NoChangeApplied;

	#[Controller]
	#[RouteBase("/uplift/theme-manager")]
	class TemplateManagerController extends BaseController{
		#[Route("GET", "/")]
		#[RequireLogin]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function themeManagerView(): Redirect|string
		{
			$currentTheme = Setting::getSettingValue(Setting::SETTING_NAMES['theme_name']);
			$faxNumbers = Setting::getSettingValue(Setting::SETTING_NAMES['companyFaxNumbers']);
			$phoneNumbers = Setting::getSettingValue(Setting::SETTING_NAMES['companyPhoneNumbers']);
			$themes = Themes::getThemes();

			// If there is no current theme, then set the current theme as the default theme
			if ($currentTheme === null){
				$currentTheme = Themes::DEFAULT_THEME_DIRECTORY_NAME;
				$currentThemeSetting = new Setting();
				$currentThemeSetting->name = Setting::SETTING_NAMES['theme_name'];
				$currentThemeSetting->value = $currentTheme;
				$currentThemeSetting->save();
			}

			// If the currently set theme no longer exists, reset it to the default theme
			if (!isset($themes[$currentTheme])){
				$currentTheme = Themes::DEFAULT_THEME_DIRECTORY_NAME;
				$currentThemeSetting = Setting::queryOne(
					columnQuery: (new ColumnQuery())
						->where("name","=",Setting::SETTING_NAMES['theme_name'])
				);
				$currentThemeSetting->value = $currentTheme;
				$currentThemeSetting->save();
			}

			if ($faxNumbers !== null){
				$faxNumbers = json_decode($faxNumbers, true);
			}else{
				$faxNumbers = [];
			}

			if ($phoneNumbers !== null){
				$phoneNumbers = json_decode($phoneNumbers, true);
			}else{
				$phoneNumbers = [];
			}

			$themeDirectory = sprintf(
				"%s/%s",
				System::THEMES_DIRECTORY,
				$currentTheme,
			);

			return Renderer::renderView(
				viewFileName: "template-manager/main.php",
				viewScope:[
					"faxNumbers"=>$faxNumbers,
					"phoneNumbers"=>$phoneNumbers,
					"currentTheme"=>$currentTheme,
					"themes"=>$themes,
					"startingThemeNavigationDirectory"=>"/",
					"themeRootDirectory"=>str_replace("\\", "/", realpath($themeDirectory)),
				],
			);
		}

		#[Route("POST", "/compile-scss-file")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function compileSCSSFile(Request $request): JSONResult
		{
			$payload = $request->getPayload();
			$currentAccount = Account::getCurrentUser();

			try {
				$filePath = $payload->getTextPayload("file-path");
			}catch(NoPayloadFound $e) {
				return new JSONError($e->getMessage());
			}

			try {
				TemplateManagerService::compileSCSSFile(
					filePath: $filePath->contents,
				);
			} catch (SassException|SassCLIException|Exceptions\AttemptToParsePartialSCSSFile|EmptyValue|FilePermissionError|IllegalDirectoryPath $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::COMPILE_SCSS->value,
				accountID: $currentAccount->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"fileCompiled"=>$filePath->contents,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("POST", "/file")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function newFile(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$parentDirectory = $payload->getTextPayload('parent-directory');
				$newFileName = $payload->getTextPayload('file-name');
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			// Optional, for when files get imported
			try {
				$fileContents = $payload->getFileUploadPayload('file-contents');
			} catch(NoPayloadFound $e) {
				// Ignoring the exception as file-contents may not always be present
				$fileContents = null;
			}

			try {
				$newFilePath = TemplateManagerService::newThemeFile(
					parentDirectory: $parentDirectory->contents,
					fileName: $newFileName->contents,
					fileContents: $fileContents?->contents,
				);
			} catch (EmptyValue|FileAlreadyExists|FilePermissionError|IllegalDirectoryPath|IllegalFileName $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::THEME_FILE_CREATED->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"filePath"=>$newFilePath,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("DELETE", "/directory")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function deleteDirectory(Request $request): JSONResult{
			$payload = $request->getPayload();
			$currentAccount = Account::getCurrentUser();

			try {
				$directory = $payload->getTextPayload("directory");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				TemplateManagerService::deleteDirectory(
					directory: $directory->contents,
				);
			} catch (IllegalDirectoryPath $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::THEME_FOLDER_DELETED->value,
				accountID: $currentAccount->id,
				ip: $request->getIP(),
				jsonData: json_encode([]),
			);

			return new JSONSuccess();
		}

		#[Route("DELETE", "/file")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function deleteThemeFile(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try{
				$filePath = $payload->getTextPayload("file-path");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				TemplateManagerService::deleteFile(
					filePath: $filePath->contents,
				);
			} catch (EmptyValue|FileDoesntExist|FilePermissionError|IllegalDirectoryPath|IllegalFileName $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::THEME_FILE_DELETED->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"filePath"=>$filePath->contents,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("GET", "/list-directory")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function getDirectoryContents(Request $request): JSONResult{
			// This will be a full path
			$directoryRequested = $request->getQueryValue("directory-requested");

			try {
				$fsDirectory = TemplateManagerService::getDirectoryContents(
					directoryPath: $directoryRequested,
				);
			} catch (DirectoryDoesntExist|IllegalDirectoryPath $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"fsDirectory"=>$fsDirectory,
				"applicationRoot"=>realpath($_SERVER['DOCUMENT_ROOT']), // So the front-end can chop this part off and create URIs
			]);
		}

		#[Route("GET", "/file-contents")]
		#[RequireLogin]
		#[UseJSON]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function getFileContents(Request $request): JSONResult{
			$filePath = $request->getQueryValue("file-path");

			try {
				$fileContents = TemplateManagerService::getFileContents(
					filePath: $filePath,
				);
			} catch (EmptyValue|FileDoesntExist|FilePermissionError|IllegalDirectoryPath $e) {
				return new JSONError($e->getMessage());
			}

			$responsePayload = new JSONSuccess([
				"fsFile"=>new FSFile(
					fileName: basename($filePath),
					fullFilePath:$filePath,
					fileSize:0,
				),
				"contents"=>$fileContents,
			]);

			// Handle binary file errors or incorrect unicode character errors
			$jsonError = json_last_error_msg();
			if ($jsonError === "Malformed UTF-8 characters, possibly incorrectly encoded"){
				return new JSONError("File cannot be sent via this API. The file contains improperly encoded UTF-8 characters. Most likely this file was double decoded or double encoded incorrectly when uploaded.");
			}

			return $responsePayload;
		}

		#[Route("POST", "/directory")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function newDirectory(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$parentDirectory = $payload->getTextPayload('parent-directory');
				$newFolderName = $payload->getTextPayload('folder-name');
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newDirectoryFullPath = TemplateManagerService::newDirectory(
					parentDirectory: $parentDirectory->contents,
					directoryName: $newFolderName->contents,
				);
			} catch (EmptyValue|FileAlreadyExists|FilePermissionError|IllegalDirectoryPath|IllegalFileName $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::THEME_FOLDER_CREATED->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"newDirectory"=>$newDirectoryFullPath,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/directory")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function renameDirectory(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$directoryPathToRename = $payload->getTextPayload('old-directory-path');
				$newFolderName = $payload->getTextPayload('new-folder-name');
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newDirectoryPath = TemplateManagerService::renameThemeDirectory(
					directory: $directoryPathToRename->contents,
					newFolderName: $newFolderName->contents,
				);
			} catch (InvalidFolderName|NewDirectoryWithSameNameExists|PathDoesntExist|DirectoryDoesntExist|EmptyValue|FileAlreadyExists|IllegalDirectoryPath|IllegalFileName $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::THEME_FOLDER_CHANGED->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"oldDirectoryPath"=>$directoryPathToRename,
					"newDirectoryPath"=>$newDirectoryPath,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/file")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function renameThemeFile(Request $request): JSONResult{
			$payload = $request->getPayload();
			$account = Account::getCurrentUser();

			try {
				$filePath = $payload->getTextPayload("file-path");
				$newFileName = $payload->getTextPayload("new-file-name");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				$newFilePath = TemplateManagerService::renameFile(
					filePath: $filePath->contents,
					newFileName: $newFileName->contents,
				);
			}catch(EmptyValue|FileAlreadyExists|FileDoesntExist|FilePermissionError|IllegalDirectoryPath|IllegalFileName|NoChangeApplied $e){
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::THEME_FILE_CHANGED->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"oldFilePath"=>$filePath->contents,
					"newFilePath"=>$newFilePath,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("PATCH", "/save-file")]
		#[RequireLogin]
		#[UseJSON]
		#[ProcessRequestBody]
		#[RequirePermission(PermissionCategories::MANAGE_TEMPLATE_FILES)]
		public function saveThemeFile(Request $request): JSONResult{
			$payload = $request->getPayload();;
			$account = Account::getCurrentUser();

			try {
				$filePath = $payload->getTextPayload("file-path");
				$fileContents = $payload->getTextPayload("file-contents");
			}catch(NoPayloadFound $e){
				return new JSONError($e->getMessage());
			}

			try {
				TemplateManagerService::saveFile(
					filePath: $filePath->contents,
					fileContents: $fileContents->contents,
				);
			} catch (EmptyValue|FileDoesntExist|FilePermissionError|IllegalDirectoryPath $e) {
				return new JSONError($e->getMessage());
			}

			ActivityLog::log(
				categoryID: ActivityLogCategories::THEME_FILE_CHANGED->value,
				accountID: $account->id,
				ip: $request->getIP(),
				jsonData: json_encode([
					"fileEdited"=>$filePath->contents,
				]),
			);

			return new JSONSuccess();
		}

		#[Route("GET", "/get-fonts")]
		#[RequireLogin]
		#[UseJSON]
		public function getFonts(): JSONResult{
			try {
				$fonts = TemplateManagerService::getActiveThemeFonts();
			} catch (Exception $e) {
				return new JSONError($e->getMessage());
			}

			return new JSONSuccess([
				"fonts"=>$fonts
			]);
		}

		#[Route("GET", "/get-fonts/fonts.css")]
		#[RequireLogin]
		#[UseJSON]
		public function getFontsCSS(): string{
			header('Content-Type: text/css');
			header('Content-Disposition: inline; filename="fonts.css"');

			return TemplateManagerService::getFontsCSS();
		}
	}