<?php
	namespace TemplateManager;

	use Exception;
	use FileSystemUtilities\exceptions\InvalidFolderName;
	use FileSystemUtilities\exceptions\NewDirectoryWithSameNameExists;
	use FileSystemUtilities\exceptions\PathDoesntExist;
	use FileSystemUtilities\FileSystemUtilities;
	use FileSystemUtilities\FSDirectory;
	use FileSystemUtilities\FSFile;
	use MonologWrapper\MonologWrapper;
	use ScssPhp\ScssPhp\Compiler;
	use ScssPhp\ScssPhp\Exception\SassException;
	use ScssPhp\ScssPhp\OutputStyle;
	use Settings\Setting;
	use System\Themes;
	use TemplateManager\Exceptions\AttemptToParsePartialSCSSFile;
	use TemplateManager\Exceptions\SassCLIException;
	use Uplift\Exceptions\DirectoryDoesntExist;
	use Uplift\Exceptions\EmptyValue;
	use Uplift\Exceptions\FileAlreadyExists;
	use Uplift\Exceptions\FileDoesntExist;
	use Uplift\Exceptions\FilePermissionError;
	use Uplift\Exceptions\IllegalDirectoryPath;
	use Uplift\Exceptions\IllegalFileName;
	use Uplift\Exceptions\NoChangeApplied;

	class TemplateManagerService{

		/**
		 * @throws EmptyValue
		 * @throws FilePermissionError
		 * @throws AttemptToParsePartialSCSSFile
		 * @throws IllegalDirectoryPath
		 * @throws SassException
		 * @throws SassCLIException
		 */
		public static function compileSCSSFile(
			string $filePath,
		): void{

			if (empty($filePath)){
				throw new EmptyValue("A file path must be provided.");
			}

			if (!Themes::isFilePathInThemeDirectory($filePath)){
				throw new IllegalDirectoryPath("The requested file must be within the current theme directories.");
			}

			$fileNameNoExtension = pathinfo($filePath, PATHINFO_FILENAME);

			// Do not allow parsing of SCSS partial files - partial files begin with an underscore
			if (str_starts_with($fileNameNoExtension, "_")){
				throw new AttemptToParsePartialSCSSFile("This API will not directly parse partial SCSS files. Only files that do not start with an underscore can be parsed directly.");
			}

			$outputFilePath = sprintf("%s/%s.css", dirname($filePath), $fileNameNoExtension);
			$outputMapFilePath = sprintf("%s/%s.css.map", dirname($filePath), $fileNameNoExtension);
			$logger = MonologWrapper::getLogger();

			// See if we can use the actual Sass binary
			if (function_exists("proc_open")){
				$command = null;
				if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
					$command = "cd sass/win-64 && sass.bat $filePath $outputFilePath --style=compressed";
				}else{
					// Linux I hope

					// If Linux, the file permissions must allow the current user to compile the file
					if (chmod("sass/linux-64/sass", 0744) === true && chmod("sass/linux-64/src/dart", 0744)){
						$command = "cd sass/linux-64 && ./sass $filePath $outputFilePath --style=compressed";
					}else{
						$logger->warning("Could not run chmod on the sass and dart files needed to allow the Linux command to run.");
					}
				}

				if ($command !== null){
					// Run the command
					$procResource = proc_open($command, [
						0 => ["pipe", "r"],
						1 => ["pipe", "w"],
						2 => ["pipe", "w"],
					], $pipes);
					fwrite($pipes[0], $command);
					fclose($pipes[0]);

					fclose($pipes[1]);

					$errMessage = stream_get_contents($pipes[2]);

					fclose($pipes[2]);
					$returnValue = proc_close($procResource);

					// A process exiting with a non-0 code is an error. Probably
					if ($returnValue !== 0){
						throw new SassCLIException($errMessage);
					}

					// We return here so that it stops the fall-through to the back-up compiler logic below (outside
					// this if statement)
					return;
				}else{
					$logger->warning("No command defined found when running SCSS compiling. Falling back to the scssphp library");
				}
			}
			// Else, fallback to the old PHP SCSS parser

			// Get relative URIs to feed to the source map options

			// Why do we not use $_SERVER['DOCUMENT_ROOT'] here?
			// Because it is not testable in unit tests - the DOCUMENT_ROOT and $_SERVER may not be
			// fully populated from CLI standpoints.
			$documentRoot = __DIR__ . "/../../..";
			$documentRootNormalized = realpath($documentRoot);
			$sourceMapFullURL = str_replace($documentRootNormalized, "", realpath($outputMapFilePath));
			$cssFileFullURL = str_replace($documentRootNormalized, "", realpath($outputFilePath));
			$sourceMapFullURL = str_replace("\\", "/", $sourceMapFullURL);
			$cssFileFullURL = str_replace("\\", "/", $cssFileFullURL);

			// Try compiling
			$compiler = new Compiler();
			$compiler->setOutputStyle(OutputStyle::COMPRESSED);
			$compiler->setImportPaths(dirname($filePath));
			$compiler->setSourceMap(Compiler::SOURCE_MAP_FILE);
			$compiler->setSourceMapOptions([
				"sourceMapURL"=>$sourceMapFullURL,
				"sourceMapFilename"=>$cssFileFullURL,
				"sourceMapBasepath"=>$_SERVER['DOCUMENT_ROOT'],
				"sourceRoot"=>"/",
			]);

			$compilationResult = $compiler->compileString(file_get_contents($filePath));

			// Write the CSS file
			$handle = @fopen($outputFilePath, "w");
			if ($handle === false){
				throw new FilePermissionError("Unable to create or open a file in the current SCSS file directory. The PHP process does not have permission to do so in this directory.");
			}
			@fwrite($handle, $compilationResult->getCss());
			@fclose($handle);

			// Write the CSS map file
			$handle = @fopen($outputMapFilePath, "w");
			if ($handle === false){
				throw new FilePermissionError("Unable to create or open a file in the current SCSS file directory. The PHP process does not have permission to do so in this directory.");
			}

			// Source map can be null here
			$sourceMapContent = $compilationResult->getSourceMap();
			if ($sourceMapContent !== null) {
				@fwrite($handle, $sourceMapContent);
			}

			@fclose($handle);
		}

		/**
		 * @param string $parentDirectory
		 * @param string $fileName
		 * @param string|null $fileContents
		 * @return string The new file path
		 * @throws FilePermissionError
		 * @throws IllegalDirectoryPath
		 * @throws EmptyValue
		 * @throws IllegalFileName
		 * @throws FileAlreadyExists
		 */
		public static function newThemeFile(
			string $parentDirectory,
			string $fileName,
			string | null $fileContents,
		): string{
			if (!Themes::isFilePathInThemeDirectory($parentDirectory)){
				throw new IllegalDirectoryPath(sprintf("Directory to create a folder in (%s) is not within the currently active theme's directory.", $parentDirectory));
			}

			if (empty($fileName)){
				throw new EmptyValue("A file name is required.");
			}

			// Is it a file safe name?
			if (!FileSystemUtilities::isNameFileSafe($fileName)){
				throw new IllegalFileName("The file name is not file-system safe. It contains invalid characters.");
			}

			$newFullFilePath = sprintf("%s/%s", $parentDirectory, $fileName);

			if (file_exists($newFullFilePath)){
				throw new FileAlreadyExists("A file with that name already exists here.");
			}

			$success = @file_put_contents($newFullFilePath, $fileContents ?? "");
			if ($success === false){
				throw new FilePermissionError("The system was unable to write a new file to this location. Most likely there is a file system permission issue. File creation failed.");
			}

			return $newFullFilePath;
		}

		/**
		 * @throws IllegalDirectoryPath
		 */
		public static function deleteDirectory(
			string $directory,
		): void{
			if (!Themes::isFilePathInThemeDirectory($directory)){
				throw new IllegalDirectoryPath(sprintf("Directory to delete (%s) is not within the currently active theme's directory.", $directory));
			}

			FileSystemUtilities::recursivelyDeleteFolder($directory);
		}

		/**
		 * @throws FileDoesntExist
		 * @throws EmptyValue
		 * @throws FilePermissionError
		 * @throws IllegalFileName
		 * @throws IllegalDirectoryPath
		 */
		public static function deleteFile(
			string $filePath,
		): void{
			if (empty($filePath)){
				throw new EmptyValue("A file path must be provided.");
			}

			// Resolve symlinks, if any
			$resolvedFilePath = realpath($filePath);

			if ($resolvedFilePath === false){
				throw new FileDoesntExist("That file doesn't exist.");
			}

			if (!Themes::isFilePathInThemeDirectory($resolvedFilePath)){
				throw new IllegalDirectoryPath("The requested file must be within the current theme directories.");
			}

			if (is_dir($resolvedFilePath)){
				throw new IllegalFileName("This API endpoint is for file deletions only.");
			}

			// Save the file
			$didSucceed = @unlink($resolvedFilePath);
			if ($didSucceed === false){
				throw new FilePermissionError("Unable to delete the file. Most likely due to OS file permissions on that file.");
			}
		}

		/**
		 * @throws DirectoryDoesntExist
		 * @throws IllegalDirectoryPath
		 */
		public static function getDirectoryContents(
			string $directoryPath,
		): FSDirectory{
			$directoryRequestedNormalized = realpath($directoryPath);

			if ($directoryRequestedNormalized === false){
				throw new DirectoryDoesntExist(sprintf(
					"The directory (%s) does not exist.",
					$directoryPath,
				));
			}

			if (!Themes::isFilePathInThemeDirectory($directoryRequestedNormalized)){
				throw new IllegalDirectoryPath(sprintf(
					"That directory (%s) is not part of the current theme",
					$directoryPath,
				));
			}

			// Sort the directories and the files
			$contents = FileSystemUtilities::fetchDirectoryContents($directoryRequestedNormalized);

			// Directory sort
			usort($contents->childDirectories, function (FSDirectory $a, FSDirectory $b){
				return strtolower($a->directoryName) > strtolower($b->directoryName) ? 1 : -1;
			});

			// File sort
			usort($contents->childFiles, function (FSFile $a, FSFile $b){
				return strtolower($a->fileNameWithoutExtension) > strtolower($b->fileNameWithoutExtension) ? 1 : -1;
			});

			return $contents;
		}

		/**
		 * @throws FileDoesntExist
		 * @throws EmptyValue
		 * @throws FilePermissionError
		 * @throws IllegalDirectoryPath
		 */
		public static function getFileContents(
			string $filePath,
		): string{
			if (empty($filePath)){
				throw new EmptyValue("A file path must be provided.");
			}

			$filePath = realpath($filePath);

			if ($filePath === false){
				throw new FileDoesntExist("The file requested does not exist.");
			}

			// Is it in the themes directory
			if (!Themes::isFilePathInThemeDirectory($filePath)){
				throw new IllegalDirectoryPath("The requested file must be within the current theme directories.");
			}

			$fileContents = "";
			// Handle large binary loads, for whatever reason
			$handle = @fopen($filePath, "r");
			if ($handle === false){
				throw new FilePermissionError("Failed to open the file. The permissions on the file may prevent PHP from opening it.");
			}

			while ($contentsRead = @fread($handle, 1024)){
				if ($contentsRead !== false) {
					$fileContents .= $contentsRead;
				}else{
					break;
				}
			}

			@fclose($handle);

			return $fileContents;
		}

		/**
		 * @throws EmptyValue
		 * @throws FilePermissionError
		 * @throws IllegalFileName
		 * @throws FileAlreadyExists
		 * @throws IllegalDirectoryPath
		 */
		public static function newDirectory(
			string $parentDirectory,
			string $directoryName,
		): string{
			if (empty($directoryName)){
				throw new EmptyValue("A new folder name is required.");
			}

			if (!Themes::isFilePathInThemeDirectory($parentDirectory)){
				throw new IllegalDirectoryPath(sprintf("Directory to create a folder in (%s) is not within the currently active theme's directory.", $parentDirectory));
			}

			// Is it a file safe name?
			if (!FileSystemUtilities::isNameFileSafe($directoryName)){
				throw new IllegalFileName("The folder name is not file-system safe. It contains invalid characters.");
			}

			$newDirectoryPath = sprintf("%s/%s", $parentDirectory, $directoryName);

			if (file_exists($newDirectoryPath)){
				throw new FileAlreadyExists("A folder with that name already exists here.");
			}

			$success = @mkdir($newDirectoryPath);
			if ($success === false){
				throw new FilePermissionError("The system was unable to call the operating system's mkdir function. Most likely there is a file system permission issue. Directory creation failed.");
			}

			return realpath($newDirectoryPath);
		}

		/**
		 * @throws NewDirectoryWithSameNameExists
		 * @throws PathDoesntExist
		 * @throws IllegalDirectoryPath
		 * @throws InvalidFolderName
		 * @throws EmptyValue
		 * @throws DirectoryDoesntExist
		 * @throws IllegalFileName
		 * @throws FileAlreadyExists
		 */
		public static function renameThemeDirectory(
			string $directory,
			string $newFolderName,
		): string{

			if (empty($directory)){
				throw new EmptyValue("Directory to rename cannot be an empty path.");
			}

			$directoryNormalized = realpath($directory);

			if ($directoryNormalized === false){
				throw new DirectoryDoesntExist("The directory $directory doesn't exist.");
			}

			if (!Themes::isFilePathInThemeDirectory($directoryNormalized)){
				throw new IllegalDirectoryPath(sprintf("Directory to rename (%s) is not within the currently active theme's directory.", $directory));
			}

			if (!FileSystemUtilities::isNameFileSafe($newFolderName)){
				throw new IllegalFileName("New folder name contains invalid/illegal characters.");
			}

			$newDirectoryPath = sprintf("%s/%s", dirname($directoryNormalized), $newFolderName);

			if (file_exists($newDirectoryPath)){
				throw new FileAlreadyExists("A folder with that name already exists here.");
			}

			FileSystemUtilities::moveDirectoryToNewLocation(
				currentDirectoryPath: $directory,
				newDirectoryPath:$newDirectoryPath,
			);

			return $newDirectoryPath;
		}

		/**
		 * @throws FileDoesntExist
		 * @throws NoChangeApplied
		 * @throws FilePermissionError
		 * @throws IllegalDirectoryPath
		 * @throws EmptyValue
		 * @throws IllegalFileName
		 * @throws FileAlreadyExists
		 */
		public static function renameFile(
			string $filePath,
			string $newFileName,
		): string{
			if (empty($filePath)){
				throw new EmptyValue("A file path must be provided.");
			}

			$filePathNormalized = realpath($filePath);

			if ($filePathNormalized === false){
				throw new FileDoesntExist("The file to rename doesn't exist.");
			}

			$currentFileName = basename($filePath);
			if ($currentFileName === $newFileName){
				// Same name. Just return the same file
				throw new NoChangeApplied("Please enter a different file name.");
			}

			if (!Themes::isFilePathInThemeDirectory($filePath)){
				throw new IllegalDirectoryPath("The requested file must be within the current theme directories.");
			}

			if (!FileSystemUtilities::isNameFileSafe($newFileName)){
				throw new IllegalFileName("New file name contains illegal characters.");
			}

			// Rename the file
			$newFilePath = sprintf("%s/%s", dirname($filePath), $newFileName);

			if (file_exists($newFilePath)){
				throw new FileAlreadyExists("A file already exists with that name in this location.");
			}

			$didSucceed = @rename($filePath, $newFilePath);
			if ($didSucceed === false){
				throw new FilePermissionError("Failed to rename the file when calling the operating system's rename command. Most likely PHP lacks permissions to edit the file system.");
			}

			return $newFilePath;
		}

		/**
		 * @throws FileDoesntExist
		 * @throws EmptyValue
		 * @throws FilePermissionError
		 * @throws IllegalDirectoryPath
		 */
		public static function saveFile(
			string $filePath,
			string $fileContents,
		): void{
			if (empty($filePath)){
				throw new EmptyValue("A file path must be provided.");
			}

			$filePathNormalized = realpath($filePath);
			if ($filePathNormalized === false){
				throw new FileDoesntExist("The file doesn't exist.");
			}

			if (!Themes::isFilePathInThemeDirectory($filePathNormalized)){
				throw new IllegalDirectoryPath("The requested file must be within the current theme directories.");
			}

			// Save the file
			$bytesWritten = @file_put_contents($filePath, $fileContents);
			if ($bytesWritten === false){
				throw new FilePermissionError("Unable to write to the file. Most likely due to OS file permissions on that file.");
			}
		}

		/**
		 * @throws Exception
		 * @return array File names, including extension, in our installed theme's font directory.
		 */
		public static function getActiveThemeFonts(): array {
			// Get the directory path of the currently active theme using the Themes class.
			$themeDirectory = Themes::getCurrentThemeDirectory();

			// Construct the full path to the fonts directory within the active theme's directory.
			$fontsDirectory = $themeDirectory . "/fonts";

			// Check if the fonts directory exists; return an empty array if it doesn't.
			if (!is_dir($fontsDirectory)) {
				return [];
			}

			// Scan the fonts directory to get an array of filenames. Exclude special entries '.' and '..'.
			$fonts = array_diff(scandir($fontsDirectory), array('..', '.'));

			// Return a re-indexed array to ensure the keys are numeric and sequential.
			return array_values($fonts);
		}

		/**
		 * Generates CSS rules for @font-face based on the active theme's fonts.
		 * @throws Exception
		 * @return string The generated CSS for embedding custom fonts.
		 */
		public static function getFontsCSS(): string {
			// Retrieve the list of active theme fonts
			$fonts = self::getActiveThemeFonts();

			// Construct the directory path for the fonts based on the current theme's name.
			$fontsDirectory = "/uplift-data/themes/" . Setting::getSettingValue(Setting::SETTING_NAMES['theme_name']) . "/fonts/";

			// Initialize an empty string to accumulate the generated CSS rules.
			$css = "";

			// Map font file extensions to their respective CSS format names.
			$fontFormats = [
				'ttf' => 'truetype',
				'otf' => 'opentype',
				'woff' => 'woff',
				'woff2' => 'woff2',
				'eot' => 'embedded-opentype',
				'svg' => 'svg'
			];

			// Loop through each font file in the list of active fonts.
			foreach($fonts as $font) {
				// Split the font filename into the base font family name and its extension.
				list($fontFamily, $fontExtension) = explode('.', $font);

				// Determine the font format based on the file extension. Default to 'unknown' if not recognized.
				$fontFormat = $fontFormats[$fontExtension] ?? 'unknown';

				// Construct the full URL to the font file based on the fonts directory and font filename.
				$fontUrl = "/uplift-data/themes/default-theme/fonts/" . $font;

				// Append the CSS rule for @font-face to the accumulated CSS string.
				$css .= "@font-face {font-family: '{$fontFamily}';src: url('{$fontUrl}') format('{$fontFormat}'),font-weight: normal;font-style: normal;}";
			}

			return $css;
		}
	}